#!/usr/bin/env python

""" Convert MARBL diagnostics file to tavg_contents

MARBL diagnostics file is a file containing a list of

    DIAGNOSTIC_NAME : frequency_operator[,frequency2frequency2_operator2, ..., frequencyN_operatorN]

POP uses this same format for defining its ecosystem-based diagnostics to allow
users to change the requested POP and MARBL diagnostics in the same place.

usage: MARBL_diags_to_tavg.py [-h] -i ECOSYS_DIAGNOSTICS_IN -d
                              MARBL_DIAGNOSTICS_LIST_IN -t TAVG_CONTENTS_OUT
                              -o MARBL_DIAGNOSTICS_OPERATOR_OUT
                              [-l LOW_FREQUENCY_STREAM]
                              [-m MEDIUM_FREQUENCY_STREAM]
                              [-g HIGH_FREQUENCY_STREAM]
                              [--lMARBL_tavg_all LMARBL_TAVG_ALL]
                              [--lMARBL_tavg_alt_co2 LMARBL_TAVG_ALT_CO2]
                              [--append APPEND]

  -i ECOSYS_DIAGNOSTICS_IN, --ecosys_diagnostics_in ECOSYS_DIAGNOSTICS_IN
                        File generated by MARBL_generate_diagnostics_file
                        (default: None)
  -d MARBL_DIAGNOSTICS_LIST_IN, --MARBL_diagnostics_list_in MARBL_DIAGNOSTICS_LIST_IN
                        File containing list of all MARBL diagnostics
                        (default: None)
  -t TAVG_CONTENTS_OUT, --tavg_contents_out TAVG_CONTENTS_OUT
                        Location of tavg_contents file to create (or append
                        to) (default: None)
  -o MARBL_DIAGNOSTICS_OPERATOR_OUT, --MARBL_diagnostics_operator_out MARBL_DIAGNOSTICS_OPERATOR_OUT
                        Location of MARBL_diagnostics_operator file to create
                        (default: None)
  -l LOW_FREQUENCY_STREAM, --low_frequency_stream LOW_FREQUENCY_STREAM
                        Stream to put low frequency output into (required if
                        not lMARBL_tavg_all) (default: 0)
  -m MEDIUM_FREQUENCY_STREAM, --medium_frequency_stream MEDIUM_FREQUENCY_STREAM
                        Stream to put medium frequency output into (required
                        if not lMARBL_tavg_all) (default: 0)
  -g HIGH_FREQUENCY_STREAM, --high_frequency_stream HIGH_FREQUENCY_STREAM
                        Stream to put high frequency output into (required if
                        not lMARBL_tavg_all) (default: 0)
  --lMARBL_tavg_all LMARBL_TAVG_ALL
                        Put all MARBL diagnostics in stream number 1 (default:
                        False)
  --lMARBL_tavg_alt_co2 LMARBL_TAVG_ALT_CO2
                        Include ALT_CO2 diagnostics in streams (default:
                        False)
  --append APPEND       Append output to tavg_contents_out instead of
                        clobbering (default: False)
"""

#######################################

def _parse_args():
    """ Parse command line arguments
    """

    import argparse

    parser = argparse.ArgumentParser(description="Generate POP tavg contents from MARBL diagnostics",
                                     formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    # Command line argument to point to MARBL diagnostics input file (required!)
    parser.add_argument('-i', '--ecosys_diagnostics_in', action='store', dest='ecosys_diagnostics_in',
                        required=True, help='File generated by MARBL_generate_diagnostics_file')

    # Command line argument to point to file containing list of MARBL diagnostics (required!)
    parser.add_argument('-d', '--MARBL_diagnostics_list_in', action='store', dest='MARBL_diagnostics_list_in',
                        required=True, help='File containing list of all MARBL diagnostics')

    # Command line argument to point to tavg_contents output file (required!)
    parser.add_argument('-t', '--tavg_contents_out', action='store', dest='tavg_contents_out',
                        required=True, help='Location of tavg_contents file to create (or append to)')

    # Command line argument to point to MARBL_diagnostics_operator output file (required!)
    parser.add_argument('-o', '--MARBL_diagnostics_operator_out', action='store', dest='MARBL_diagnostics_operator_out',
                        required=True, help='Location of MARBL_diagnostics_operator file to create')

    # Command line arguments for the different streams to use (low, medium, high)
    parser.add_argument('-l', '--low_frequency_stream', action='store', dest='low_frequency_stream',
                        type=int, default= 0, help='Stream to put low frequency output into (required if not lMARBL_tavg_all)')

    parser.add_argument('-m', '--medium_frequency_stream', action='store', dest='medium_frequency_stream',
                        type=int, default= 0, help='Stream to put medium frequency output into (required if not lMARBL_tavg_all)')

    parser.add_argument('-g', '--high_frequency_stream', action='store', dest='high_frequency_stream',
                        type=int, default= 0, help='Stream to put high frequency output into (required if not lMARBL_tavg_all)')

    # Should all MARBL diagnostics be included in the first tavg stream?
    parser.add_argument('--lMARBL_tavg_all', action='store', dest='lMARBL_tavg_all',
                        type=bool, default=False, help="Put all MARBL diagnostics in stream number 1")

    # Should MARBL's ALT_CO2 diagnostics be included in the tavg streams?
    parser.add_argument('--lMARBL_tavg_alt_co2', action='store', dest='lMARBL_tavg_alt_co2',
                        type=bool, default=False, help="Include ALT_CO2 diagnostics in streams")

    # Should lines be appended to tavg_contents_out instead of clobbering?
    parser.add_argument('--append', action='store', dest='append', default=False,
                        type=bool, help='Append output to tavg_contents_out instead of clobbering')

    return parser.parse_args()

#######################################

def _parse_line(line_in):
    """ Take a line of input from the MARBL diagnostic output and return the variable
        name, frequency, and operator. Lines that are commented out or empty should
        return None for all three; non-empty lines that are not in the proper format
        should trigger errors.

        If they are not None, frequency and operator are always returned as lists
        (although they often have just one element).
    """
    import sys

    line_loc = line_in.split('#')[0].strip()
    # Return None, None if line is empty
    if len(line_loc) == 0:
        return None, None, None

    logger = logging.getLogger("__name__")
    line_split = line_loc.split(':')
    if len(line_split) != 2:
        logger.error("Can not determine variable name from following line: '%s'" % line_in)
        sys.exit(1)

    freq = []
    op = []
    for freq_op in line_split[1].split(','):
        freq_op_split = freq_op.strip().split('_')
        if len(freq_op_split) != 2:
            logger.error("Can not determine frequency and operator from following entry: '%s'" % line_split[1])
            sys.exit(1)
        freq.append(freq_op_split[0])
        op.append(freq_op_split[1])

    return line_split[0].strip(), freq, op

#######################################

def _get_freq(frequency, frequency_dict):
    """ Convert MARBL frequency ('never', 'low', 'medium', 'high') to POP streams.
        Note that frequency_dict must be a dictionary with keys low, medium, and
        high containing the associated stream number for that frequency
    """
    freq_loc = frequency.strip().lower()
    if freq_loc == 'never':
        return '#'
    try:
        return '%d' % frequency_dict[freq_loc]
    except:
        logger = logging.getLogger("__name__")
        logger.error("Unrecognized output frequency: '%s'" % freq_loc)

#######################################

def diagnostics_to_tavg_and_operators(ecosys_diagnostics_in,
                                      MARBL_diagnostics_list_in,
                                      tavg_contents_out,
                                      MARBL_diagnostics_operator_out,
                                      append,
                                      lMARBL_tavg_all,
                                      lMARBL_tavg_alt_co2,
                                      frequency_dict):

    import os, sys, logging
    logger = logging.getLogger("__name__")
    labort = False
    processed_vars = []

    # 1. Check arguments:
    #    ecosys_diagnostics_in can not be None and must be path of an existing file
    if ecosys_diagnostics_in == None:
        logger.error("Must specific ecosys_diagnostics_in")
        labort = True
    elif not os.path.isfile(ecosys_diagnostics_in):
        logger.error("File not found %s" % ecosys_diagnostics_in)
        labort = True
    #    MARBL_diagnostics_list_in can not be None and must be path of an existing file
    if MARBL_diagnostics_list_in == None:
        logger.error("Must specific MARBL_diagnostics_list_in")
        labort = True
    elif not os.path.isfile(MARBL_diagnostics_list_in):
        logger.error("File not found %s" % MARBL_diagnostics_list_in)
        labort = True
    if labort:
        sys.exit(1)

    # 2. Read list of all MARBL diagnostics
    MARBL_diagnostics_list = []
    with open(MARBL_diagnostics_list_in, 'r') as file_in:
        MARBL_diagnostics_list = [diag.strip() for diag in file_in.readlines()]

    # Open files for output
    try:
        if append:
            tavg_file = open(tavg_contents_out, 'a')
        else:
            tavg_file = open(tavg_contents_out, 'w')
    except:
        logger.error("Can not open %s to write" % tavg_contents_out)
        sys.exit(1)
    try:
        diagnostics_operator_file = open(MARBL_diagnostics_operator_out, 'w')
    except:
        logger.error("Can not open %s to write" % MARBL_diagnostics_operator_out)
        sys.exit(1)

    # Files header
    tavg_file.write('#  The following diagnostics were provided via\n')
    tavg_file.write('#  %s\n' % ecosys_diagnostics_in)
    diagnostics_operator_file.write('#  The following operators were requested via\n')
    diagnostics_operator_file.write('#  %s\n' % ecosys_diagnostics_in)

    # Read ecosys_diagnostics_in line by line, convert each line to tavg content
    with open(ecosys_diagnostics_in, 'r') as file_in:
        all_lines = file_in.readlines()

    for line in all_lines:
        varname, frequency, operator = _parse_line(line.strip())
        # Continue to next line in the following circumstances
        # i.  varname = None
        if varname == None:
            continue
        # ii. Skip ALT_CO2 vars unless explicitly requested
        if (not lMARBL_tavg_alt_co2) and ("ALT_CO2" in varname):
            continue

        # Abort if varname has already appeared in file
        if varname in processed_vars:
            logger.error("'%s' appears in %s multiple times" % (varname, ecosys_diagnostics_in))
            sys.exit(1)
        processed_vars.append(varname)

        # tavg_contents
        for n, freq in enumerate(frequency):
            if lMARBL_tavg_all:
                use_freq = '1'
            else:
                use_freq = _get_freq(freq, frequency_dict)
            if n == 0:
                tavg_file.write('%s  %s\n' % (use_freq, varname))
            else:
                tavg_file.write('%s  %s_%d\n' % (use_freq, varname, n+1))

        # MARBL_diagnostics_operator
        if varname in MARBL_diagnostics_list:
            if lMARBL_tavg_all:
                use_op = ', '.join(['average']*len(operator))
            else:
                use_op = ', '.join(operator)
            if use_op != 'none':
                diagnostics_operator_file.write('%s : %s\n' % (varname, use_op))

    # File footer
    tavg_file.write('#  end of diagnostics from\n# %s\n' % ecosys_diagnostics_in)

    # Close streams
    tavg_file.close()
    diagnostics_operator_file.close()

#######################################

if __name__ == "__main__":
    # Parse command line arguments
    args = _parse_args()
    frequency_dict = dict()
    frequency_dict['low'] = args.low_frequency_stream
    frequency_dict['medium'] = args.medium_frequency_stream
    frequency_dict['high'] = args.high_frequency_stream

    import logging
    logging.basicConfig(format='%(levelname)s (%(funcName)s): %(message)s', level=logging.DEBUG)

    # call diagnostics_to_tavg()
    diagnostics_to_tavg_and_operators(args.ecosys_diagnostics_in,
                                      args.MARBL_diagnostics_list_in,
                                      args.tavg_contents_out,
                                      args.MARBL_diagnostics_operator_out,
                                      args.append,
                                      args.lMARBL_tavg_all,
                                      args.lMARBL_tavg_alt_co2,
                                      frequency_dict)
