"""
This is the new way of selecting latbox and lonbox of the ERA interim version 2 
reanalysis data set. Selected by finding the index of the TempestTrack lat and 
lon points and selecting a box that is 7 grid points below and 7 gridpoints above 
the center in either direction for a latbox and lonbox that is 15 X 15 gridpoints,
or approximately 9.8 degrees.
"""
import numpy as np 
import pandas as pd
import xarray as xr
import sys

# define the filepath of where to find the tracks
tempest_filepath = r'/home/awing/ibtracs/trajectories.txt.ERAI'

def boxavg(thing,lat,lon):
    """
    Calculate cosine(latitude)-weighted box average 
    e.g., h_avg=boxavg(h,lat)
    Important: h and lat must already be restricted to box of interest
    """
    
#       Define weights, *(idata*0+1) magically repeats it to
#       match dimensions of idata
    # wgt = np.cos(np.deg2rad(lat))*(thing*0+1)
    # thing_avg = (wgt*thing).sum()/wgt.sum() #sum over all grid points in box
    coslat_values=np.transpose(np.tile(np.cos(np.deg2rad(lat)),(len(lon),1)))
    thing1 = thing*coslat_values
    thing2 = thing1/thing1
    average=np.nansum(np.nansum(thing1,0))/np.nansum(np.nansum(coslat_values*thing2,0))

    return average

start_year = 1980 # beginning of most TempestExtremes tracks is 1980
end_year = 2016 # depends on how far the dataset goes
#----------Get the data organized to ultimately group by storm-----------#


df = pd.read_csv(tempest_filepath,sep='\t',header=None,names=\
                 ['flag','num_timesteps','dummy','lon','lat','pressure (hPa)',\
                  'windspeed (m/s)','dummy2','year','month','day','hour'])
# drop columns we don't need
df = df.drop(columns=(['dummy','dummy2'])) 
# add in a flag so that it knows when to 
df_starts = df[df['flag']=='start']
# start with assigning a stormID of 1 for the dataset 
storm_id = 1
# loop through each storm at a time and add in the appropriate storm_ids
for idx, num_steps in zip(df_starts.index, df_starts['num_timesteps'].values):
    # add in a column for the storm ID, arbitrary for each dataset
    df.loc[idx:idx+num_steps+1,'stormid'] = storm_id 
    # add 1 to the stormID each time you get to the end of a particular storm\
    # track to continue the loop
    storm_id += 1   

# identify and remove the flag column and num_timesteps column since we've \
# already added in the stormIDS
df = df[df['flag']!='start']
df = df.drop(columns=(['flag','num_timesteps'])).reset_index(drop=True)

# do some formatting to make some of the column headings formatted the way I want them
df.loc[:,'pressure (hPa)'] = df.loc[:,'pressure (hPa)'] / 100.
df.loc[:,'year'] = df.loc[:,'year'].astype(int).astype(str)
df.loc[:,'month'] = df.loc[:,'month'].astype(int).astype(str)
df.loc[:,'day'] = df.loc[:,'day'].astype(int).astype(str)
df.loc[:,'hour'] = df.loc[:,'hour'].astype(int).astype(str)
df.loc[:,'windspeed (m/s)'] = df.loc[:,'windspeed (m/s)']
df.loc[:,'hour'] = np.where(df['hour'].astype(int)<10,'0'+ df['hour'], df['hour'])

# create a timestamp for each time step so that it is easier to read in later
df.loc[:,'datetime'] = pd.to_datetime(df['year'] + ' ' + df['month'] + ' ' + df['day'] + ' ' + df['hour'])

#------------Now create a list of storm ID's by year for looping through\
#    and matching to the data----------#

ibtracs_by_year = {year: [] for year in range(start_year,end_year+1)} # make \
#an empty array of storm tracks by year, where you add 2 just in case

for storm in range(1,1593): # range of storms to loop through to generate \
#    list of storms by year. Note I manually found the last stormID
    sys.stdout.write(f'\r{storm}/1593')
    # show the progress in the command line
    
    # get the list of characteristics for the storm ID you're currently on
    ds_storm = df[df['stormid']==storm]
    # get the unique list of times for that storm
    times = ds_storm['datetime'].values
    # do some conversion to make it easier to work with
    times = pd.to_datetime(times) # convert pandas time to datetimeindex
    times = times.to_pydatetime() # convert to datetime.datetime objects
    # only include times for which the dataset is valid
    if (times[0].year < start_year and times[-1].year < start_year) or (\
    times[0].year > end_year and times[-1].year > end_year): 
        continue
    second_part_flag = False
    if times[0].year != times[-1].year: # this is to check if a storm starts in one year and ends in another, testing the first and last time in the storm
        ibtracs_by_year[times[-1].year].append(storm) # append the extra storm to the list to ensure all times are included in the year
    ibtracs_by_year[times[0].year].append(storm) # append the list of storms to the correct year, including ones that have parts of two years

# make a list to loop through each year--not the most efficient but I did this a long time ago haha
year_list=[] # make a list of years
for key in ibtracs_by_year:
    year_list.append(key)

"""-----------------------------------------------------------------------------------
This is where we start our main loops to extract all the data ------------------------
-----------------------------------------------------------------------------------"""
print('\n')
for current_year in range(1980,2017):

    storm_times=[] # create an empty list to append to of times for all storms in the current year
    #outside the storm loop but inside the year loop rewrite the SEF as a function of lat and lon box
    
    #define the NaN arrays for various variable anomalies. don't need averages. also and boxlon
    # save both the variable and the variable anomaly, write 2 files, one w regular vars and one w anomalies
    #will need len(ib(current))
    
    #to find nlat and nlon I printed len(lat) and len(lon) after the conditionals and found the max value
    nlat = 15
    nlon = 15
    
    # nstorms = number of storms in that year --> in this case it is 41
    nstorms = len(ibtracs_by_year[current_year])
    # print(nstorms)
    
    #ntracks is maximum possible number of "observations" along the track of a storm
    #in this case the max num_timesteps is 86
    ntracks = max(df_starts['num_timesteps'])   
    # print(ntracks)
    
    #We need to make an empty 4D array of NaNs that have the dimensions lat x lon x nstorms x ntracks
    #want to store the regular variable and the box anomaly of net SW, net LW, and , and all those with hanom
    SWanomsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SWanomsave[:] = np.nan
    LWanomsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    LWanomsave[:] = np.nan
    SEFanomsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SEFanomsave[:] = np.nan
    SWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SWsave[:] = np.nan
    LWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    LWsave[:] = np.nan
    SEFsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SEFsave[:] = np.nan
    
    boxlat = np.empty((nlat,nstorms,ntracks))
    boxlon = np.empty((nlon,nstorms,ntracks))
    boxlat[:] = np.nan
    boxlon[:] = np.nan
    
    yearsave = np.empty((nstorms, ntracks))
    yearsave[:] = np.nan   
    monthsave = np.empty((nstorms, ntracks))
    monthsave[:] = np.nan  
    daysave = np.empty((nstorms, ntracks))
    daysave[:] = np.nan    
    hoursave = np.empty((nstorms, ntracks))
    hoursave[:] = np.nan
    
    windsave = np.empty((nstorms, ntracks))
    windsave[:] = np.nan 
    
    hsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hsave[:] = np.nan
    
    varhsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    varhsave[:] = np.nan
    
    hSWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hSWsave[:] = np.nan
    hLWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hLWsave[:] = np.nan   
    hSEFsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hSEFsave[:] = np.nan
    
    centerlatsave = np.empty((nstorms,ntracks))
    centerlatsave[:] = np.nan
    centerlonsave = np.empty((nstorms,ntracks))
    centerlonsave[:] = np.nan
    
    
    for s, storm in enumerate(ibtracs_by_year[current_year]): # start looping through every storm 
        # print(s,storm)
        sys.stdout.write(f'\r{s}/{nstorms}')
        # generate a list of times valid for the whole year

        # this is a repeat of above where you're just formatting the times and making them how you want to be
        ds_storm = df[df['stormid']==storm]
        time_list = [tstamp.to_pydatetime() for tstamp in pd.to_datetime(ds_storm['datetime'].values)] # make a timelist and convert to datetime.datetime
        time_list = [t.replace(second=0, microsecond=0) for t in time_list]   # Round to nearest minute to avoid weird rounding errors
        storm_times.append(time_list) # append the time_list for each storm to the overall storm_times list altogether

        # Import ibtracs variables
        track_lat = ds_storm['lat'].values
        track_lon = ds_storm['lon'].values
        sid = ds_storm['stormid'].values # import storm ID values from USA dataset
        wind = ds_storm['windspeed (m/s)'].values # import wind speeds from USA dataset
        mslp = ds_storm['pressure (hPa)'].values # import mean sea level pressure from USA dataset in mb

        #################################################################################################

        # Loop through each time and assign the flux variables to the TC
        
        #################################################################################################
     
        time_datasets=[] # empty array of datasets by time        
        used_times = [] # list of times used

        for t, time in enumerate(time_list):
       
#             if time_list[t].year != current_year: # don't accumulate precip for time steps in years other than current one
# #*** might not need
#                 continue
            
            dslat = xr.open_dataset(r'/huracan/tank2/columbia/reanalysis/erai/2D/flux/erai.flux.2005100100.nc')
            lat = dslat.g4_lat_2
            lon = dslat.g4_lon_3
            idxlat = pd.Index(lat)
            idxlon = pd.Index(lon)
            latlist = idxlat.tolist()
            lonlist = idxlon.tolist()
            
            searchlat = dslat.g4_lat_2.sel(g4_lat_2 = track_lat[t], method = 'nearest')
            ilat = latlist.index(searchlat)
            searchlon = dslat.g4_lon_3.sel(g4_lon_3 = track_lon[t], method = 'nearest')
            ilon = lonlist.index(searchlon)
             
            ilat_up = ilat + 8
            ilat_down = ilat - 7
            ilon_up = ilon + 8
            ilon_down = ilon - 7
        
        # use the defined box indices to select the actual values of the lat and lon list that
        # correspond to those indices     
            latbox = dslat.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
            lonbox = dslat.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
            
            centerlat = track_lat[t]
            centerlon = track_lon[t]
            
            # get the formats of the dates and stuff the way you want them
            ym = time.strftime('%Y%m') # convert the year month time from datetime.datetime to a string
            ym = str(ym) # make the ym a string to be able to compare to the file name
            year = time.strftime('%Y') # convert the year from datetime.datetime to a string
            month = time.strftime('%m') # convert the month to a string
            month = month.zfill(2) #this adds a floating zero in front of the month
            day = time.strftime('%d')
            hour = time.strftime('%H') # convert the hour to a string
            m_minus1 = int(month)-1
            m_minus1 = str(m_minus1)
            m_minus1 = m_minus1.zfill(2) #this adds a floating zero in front of the month if it is one digit
            y_minus1 = int(year)-1
            y_minus1 = str(y_minus1)
            d_minus1 = int(day)-1
            d_minus1 = str(d_minus1)
            d_minus1 = d_minus1.zfill(2) #adds a floating zero if one digit
            MM = month.zfill(2)
            HH = hour.zfill(2)
            DD = day.zfill(2)
            YYYY = year
            MM_1 = m_minus1
            DD_1 = d_minus1
            YYYY_1 = y_minus1
     
            """
            The following conditional statements are to organize which datasets to open/concatenate
            depending on which day, month, and time it is
         
            00ZD = 00ZD - 21Z(D-1) + 03ZD
            06ZD = 09ZD - 03ZD
            12ZD = 12ZD - 09ZD + 15ZD
            18ZD = 21ZD - 15ZD
            """
            dataset_filepath = r'/huracan/tank2/columbia/reanalysis/erai/2D/flux/' #version 2 of the reanalysis erai
# ------------------------------------------------------------------------------------------------ 
# conditional statements to read in the variables depending on which day, month, and time it is.
# Option 1: 00Z AND NOT the 1st of the month
# Option 2: 12Z any day
# Option 3: 6Z any day
# Option 4: 18Z any day
# Option 5: 00Z AND 1st of the month AND (months 2 OR 4 OR 6 OR 8 OR 9 OR 11), months where the previous month has 31 days
# Option 6: 00Z AND 1st of the month AND (months 5 OR 7 OR 10 OR 12), months where the previous month has 30 days
# Option 7: 00Z AND January 1st
# Option 8: 00Z AND 1st of the month AND March AND (years 1980 OR 1984 OR 1988 OR 1992 OR 1996 OR 2000 OR 2004 OR 2008 OR 2012 OR 2016), leap years 
# Else (Option 9): 00Z March 1st during non leap years 
# ------------------------------------------------------------------------------------------------        
            
            
# Option 1            
            #for when it is 00Z AND NOT the 1st of the month
            if (time_list[t].day != 1 and time_list[t].hour == 0):
                  
                # filename_flux = '%s-01.nc' % (year)
                filename_flux = 'erai.flux.%s%s%s00.nc' % (YYYY,MM,DD)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
                
                filename_flux2 = 'erai.flux.%s%s%s12.nc' % (YYYY,MM,DD_1)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours')
           
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                #both of these will contribute to SEF
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of the first ds, 3Z DD
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #last element of the 2nd ds, 00Z DD 
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #2nd to last element of ds2, 21Z DD-1 
                shf = shf2 - shf3 + shf1
              
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
             
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close
                
# Option 2            
            #now for when it is 12Z any day
            elif time_list[t].hour == 12:
                filename_flux = 'erai.flux.%s%s%s12.nc' % (YYYY,MM,DD)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
                
                filename_flux2 = 'erai.flux.%s%s%s00.nc' % (YYYY,MM,DD)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours') 
            
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1,15Z DD
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #last element of ds2, 12Z DD
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #2nd to last element of ds2, 9Z DD
                shf = shf2 - shf3 + shf1
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc SW to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close
            
# Option 3
            # 6Z at any day    
            elif time_list[t].hour == 6:
                filename_flux = 'erai.flux.%s%s%s00.nc' % (YYYY,MM,DD)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
                
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1, 3Z DD
                shf2 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #third element of ds1, 9Z DD
                shf = shf2 - shf1 
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf2 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                lhf = lhf2 - lhf1 
                
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                ssw = ssw2 - ssw1 
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                tsw = tsw2 - tsw1 
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tlw2 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
 
                ds1.close

# Option 4                
            # 18Z at any day    
            elif time_list[t].hour == 18:
                filename_flux = 'erai.flux.%s%s%s12.nc' % (YYYY,MM,DD)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
              
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1, 15Z DD
                shf2 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #third element of ds1, 21Z DD
                shf = shf2 - shf1 
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf2 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                lhf = lhf2 - lhf1 
                
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                ssw = ssw2 - ssw1 
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                tsw = tsw2 - tsw1 
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tlw2 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
 
                ds1.close
#Option 5                
            #00Z AND 1st of the month AND (months 2 OR 4 OR 6 OR 8 OR 9 OR 11), months where the previous month has 31 days
            elif (time_list[t].month == 2 or time_list[t].month == 4 or time_list[t].month == 6 or \
                  time_list[t].month == 8 or time_list[t].month == 9 or time_list[t].month == 11) and\
                  time_list[t].day == 1 and time_list[t].hour == 0:
                      
                filename_flux = 'erai.flux.%s%s%s00.nc' % (YYYY,MM,DD)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
                
                filename_flux2 = 'erai.flux.%s%s3112.nc' % (YYYY,MM_1)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours')
          
                #sensible heat flux 
                # shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up),g4_lon_3 = slice(ilon_down,ilon_up))  

                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                shf = shf2 - shf3 + shf1
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc SW to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))  
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close
#Option 6                
            #00Z AND 1st of the month AND (months 5 OR 7 OR 10 OR 12), months where the previous month has 30 days
            elif (time_list[t].month == 5 or time_list[t].month == 7 or time_list[t].month == 10 or \
                  time_list[t].month == 12) and time_list[t].day == 1 and time_list[t].hour == 0:
                      
                filename_flux = 'erai.flux.%s%s%s00.nc' % (YYYY,MM,DD)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
               
                filename_flux2 = 'erai.flux.%s%s3012.nc' % (YYYY,MM_1)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours')
              
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1, 3Z DD
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #last element of ds2, 00Z DD
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #2nd to last element of ds2, 21Z DD-1
                shf = shf2 - shf3 + shf1
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc SW to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close  
#Option 7
            #00Z AND January 1st        
            elif time_list[t].month == 1 and time_list[t].day == 1 and time_list[t].hour == 0:
                filename_flux = 'erai.flux.%s010100.nc' % (YYYY)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
             
                filename_flux2 = 'erai.flux.%s123112.nc' % (YYYY_1)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours')
              
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1,3Z DD
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #last element of ds2, 00Z DD
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #2nd to last element of ds2, 21Z DD-1
                shf = shf2 - shf3 + shf1
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc SW to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close                
#Option 8: Leap years on the first day of march at 00Z
            elif (time_list[t].year % 4 == 0) and time_list[t].month == 3 and\
                  time_list[t].day == 1 and time_list[t].hour == 0:
                      
                filename_flux = 'erai.flux.%s030100.nc' % (YYYY)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')

                
                filename_flux2 = 'erai.flux.%s022912.nc' % (YYYY)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours')
                
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1,3Z DD
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #last element of ds2, 00Z DD
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #2nd to last element of ds2, 21Z DD-1
                shf = shf2 - shf3 + shf1
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc SW to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close         
#Option 9: non leap years on the first day of march at 00Z
            else:
                # print('\n')
                # print(YYYY, MM, DD, HH)
                filename_flux = 'erai.flux.%s030100.nc' % (YYYY)
                ds1 = xr.open_dataset(dataset_filepath + filename_flux)
                ds1 = ds1.squeeze('initial_time0_hours')
                
                filename_flux2 = 'erai.flux.%s022812.nc' % (YYYY)
                ds2 = xr.open_dataset(dataset_filepath + filename_flux2)
                ds2 = ds2.squeeze('initial_time0_hours')
               
                #sensible heat flux 
                shf1 = ds1.SSHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #first element of ds1,3Z DD
                shf2 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #last element of ds2, 00Z DD
                shf3 = ds2.SSHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #2nd to last element of ds2, 21Z DD-1
                shf = shf2 - shf3 + shf1
                
                #latent heat flux
                lhf1 = ds1.SLHF_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                lhf2 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf3 = ds2.SLHF_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                lhf = lhf2 - lhf3 + lhf1
                
                #need TOA and sfc SW to get net SW  
                #ssw = surface solar radiation (SW)
                ssw1 = ds1.SSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw2 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) #
                ssw3 = ds2.SSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                ssw = ssw2 - ssw3 + ssw1
                
                #tsw = TOA sw radiation 
                tsw1 = ds1.TSR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw2 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up))
                tsw3 = ds2.TSR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tsw = tsw2 - tsw3 + tsw1
                
                #need TOA and sfc LW to get net LW  
                #slw = surface thermal radiation (LW)
                slw1 = ds1.STR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw2 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw3 = ds2.STR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                slw = slw2 - slw3 + slw1
                
                #tlw = TOA LW radiation 
                tlw1 = ds1.TTR_GDS4_SFC.isel(forecast_time1 = 0,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw2 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -1,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw3 = ds2.TTR_GDS4_SFC.isel(forecast_time1 = -2,g4_lat_2 = slice(ilat_down,ilat_up), g4_lon_3 = slice(ilon_down,ilon_up)) 
                tlw = tlw2 - tlw3 + tlw1
                
                lat = ds1.g4_lat_2.isel(g4_lat_2 = slice(ilat_down,ilat_up))
                lon = ds1.g4_lon_3.isel(g4_lon_3 = slice(ilon_down,ilon_up))
                
                SW = tsw - ssw
                LW = tlw - slw  
                #net shortwave flux converted to W/m2
                SW = SW/(3600*6)
                #net longwave flux converted to W/m2
                LW = LW/(3600*6)
                
                #surface enthalpy flux
                SEF = shf + lhf
                #convert to W/m2, change sign convention
                SEF = -SEF/(3600*6)
                # print(SEF)
                
                ds1.close
                ds2.close          
#--------------------------------------------------------------------------------------#
#                                 THIS ENDS THE CONDITIONALS                           #
#--------------------------------------------------------------------------------------#
            #still inside innermost loop over storms but out of the conditonal
            #read in h from huracan 
            #now define anoms with the 4D indices    
            hpath = '/huracan/tank2/columbia/reanalysis/erai/2D/h/'
            hfilename = 'erai.h.%s%s%s%s.nc' % (YYYY,MM,DD,HH)
            dsh = xr.open_dataset(hpath + hfilename)
            hlat = dsh.latitude.isel(latitude = slice(ilat_down,ilat_up))
            h = dsh.h.isel(latitude = slice(ilat_down,ilat_up), longitude = slice(ilon_down,ilon_up))

            #landmask stuff
            dslm = xr.open_dataset(r'/home/cdirkes/eraint/landmask/land_sea_mask.nc')
            lm = dslm.lsm.isel(time = 0)#, latitude = slice(ilat_down,ilat_up), longitude = slice(ilon_down,ilon_up))
            lonlm = dslm.longitude #ead in longitude variable from that dataset
            latlm = dslm.latitude #read in latitude variable from that dataset
            idxlatlm = pd.Index(latlm)
            idxlonlm = pd.Index(lonlm)
            latlistlm = idxlatlm.tolist()
            lonlistlm = idxlonlm.tolist() 
            lsm = np.zeros((nlat,nlon))
            for a in range(0,len(latbox)):
                for o in range(0,len(lonbox)):
                    testlat = np.array(latbox[a])
                    testlon = np.array(lonbox[o])
                    
                    searchlatlm = np.array(dslm.latitude.sel(latitude = testlat, method = 'nearest'))
                    # print(a, testlat, np.array(searchlat),np.array(searchlatlm))
                    lmlat = latlistlm.index(searchlatlm)
                    
                    searchlonlm = np.array(dslm.longitude.sel(longitude = testlon, method = 'nearest'))
                    # print(o, testlon, np.array(searchlon),np.array(searchlonlm))
                    lmlon = lonlistlm.index(searchlonlm)
                    if lm[lmlat,lmlon] > 0.2:
                        lsm[a,o] = np.nan   #(20 x nstorms x ntracks)
   
            lsm = np.array(lsm)
            # print(lsm)
            h = np.array(h)
            SEF = np.array(SEF)
            SW = np.array(SW)
            LW = np.array(LW)
            
            if (np.shape(lsm) != np.shape(SW)) or (np.shape(lsm) != np.shape(h)) or (np.shape(lsm) != np.shape(LW)) or (np.shape(lsm) != np.shape(SEF)):
                lsm = lsm[:,0:(np.size(lonbox))]             
            # print(np.isnan(SEF).all())

            h = h + lsm
            SEF = SEF + lsm
            SW = SW + lsm
            LW = LW + lsm
            
            havg = boxavg(h, latbox, lonbox)
            SWavg = boxavg(SW, latbox, lonbox)
            LWavg = boxavg(LW, latbox, lonbox)
            SEFavg = boxavg(SEF, latbox, lonbox) 
                
            hanom = h - havg
            hsave[0:len(latbox),0:len(lonbox),s,t] = h
            varhsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(hanom))
                                     
            SWanom = SW - SWavg
            SWanomsave[0:len(latbox),0:len(lonbox),s,t] = SW - SWavg    
               
            LWanom = LW - LWavg
            LWanomsave[0:len(latbox),0:len(lonbox),s,t] = LW - LWavg
            
            SEFanom = SEF - SEFavg
            SEFanomsave[0:len(latbox),0:len(lonbox),s,t] = SEF - SEFavg
            
            windsave[s,t] = wind[t]

            boxlat[0:len(latbox),s,t] = latbox
            boxlon[0:len(lonbox), s, t] = lonbox
            
            centerlatsave[s,t] = track_lat[t]
            centerlonsave[s,t] = track_lon[t]
             #change len to latbox and longbox    
            SWsave[0:len(latbox),0:len(lonbox),s,t] = SW
            LWsave[0:len(latbox),0:len(lonbox),s,t] = LW
            SEFsave[0:len(latbox),0:len(lonbox),s,t] = SEF
            
            hSWsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(SWanom))
            hLWsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(LWanom))
            hSEFsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(SEFanom))
                     
            yearsave[s,t] = time_list[t].year
            monthsave[s,t] = time_list[t].month
            daysave[s,t] = time_list[t].day
            hoursave[s,t] = time_list[t].hour
            
            # print(np.shape(hsave), np.shape(SWsave), np.shape(SWsave))
                        
            #convert them all into xarrays
            SWsave = xr.core.dataarray.DataArray(SWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            LWsave = xr.core.dataarray.DataArray(LWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            SEFsave = xr.core.dataarray.DataArray(SEFsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            
            SWanomsave = xr.core.dataarray.DataArray(SWanomsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            LWanomsave = xr.core.dataarray.DataArray(LWanomsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            SEFanomsave = xr.core.dataarray.DataArray(SEFanomsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            
            hsave = xr.core.dataarray.DataArray(hsave, dims=['nlat', 'nlon', 'nstorms', 'ntracks'])
            varhsave = xr.core.dataarray.DataArray(varhsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            hSWsave = xr.core.dataarray.DataArray(hSWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            hLWsave = xr.core.dataarray.DataArray(hLWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            hSEFsave = xr.core.dataarray.DataArray(hSEFsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])

            
            windsave = xr.core.dataarray.DataArray(windsave, dims=['nstorms', 'ntracks'])

            boxlat = xr.core.dataarray.DataArray(boxlat, dims=['nlat', 'nstorms','ntracks'])
            boxlon = xr.core.dataarray.DataArray(boxlon, dims=['nlon', 'nstorms','ntracks'])
            centerlatsave = xr.core.dataarray.DataArray(centerlatsave, dims=['nstorms','ntracks'])
            centerlonsave = xr.core.dataarray.DataArray(centerlonsave, dims=['nstorms','ntracks'])
            
            yearsave = xr.core.dataarray.DataArray(yearsave, dims=['nstorms', 'ntracks'])
            monthsave = xr.core.dataarray.DataArray(monthsave, dims=['nstorms', 'ntracks'])
            daysave = xr.core.dataarray.DataArray(daysave, dims=['nstorms', 'ntracks'])
            hoursave = xr.core.dataarray.DataArray(hoursave, dims=['nstorms', 'ntracks'])
            
    """
    Write variable (already defined as a data array) to a netcdf file. making sure that it
    is within the first nested loop over all the years but not within the storms or tracks
    Need: SEF, SW, LW, all their anomalies, and lat and lon
    """
    #save all the units and attributes of each variable
    yearsave.attrs['units'] = 'year of specific storm'
    yearsave.attrs['long_name'] = 'year'
    monthsave.attrs['units'] = 'month of specific storm'
    monthsave.attrs['long_name'] = 'month'
    daysave.attrs['units'] = 'day of specific storm'
    daysave.attrs['long_name'] = 'day'
    hoursave.attrs['units'] = '6hr accumulation centered around hour'
    hoursave.attrs['long_name'] = 'hour, UTC'

    SWsave.attrs['units']='W/m^2'
    SWsave.attrs['long_name']='Net shortwave flux'
    SWsave.attrs['_FillValue']=-9999
    SWsave.attrs['GridType']='Gaussian Grid'
    
    LWsave.attrs['units']='W/m^2'
    LWsave.attrs['long_name']='Net longwave flux'
    LWsave.attrs['_FillValue']=-9999
    LWsave.attrs['GridType']='Gaussian Grid'
  
    SEFsave.attrs['units']='W/m^2'
    SEFsave.attrs['long_name']='Surface enthalpy flux'
    SEFsave.attrs['_FillValue']=-9999
    SEFsave.attrs['GridType']='Gaussian Grid'
    
    hsave.attrs['units']='J/m^2'
    hsave.attrs['long_name']='column integrated moist static energy'
    hsave.attrs['_FillValue']=-9999
    hsave.attrs['GridType']='Gaussian Grid'
    
    varhsave.attrs['units']='J^2/m^4'
    varhsave.attrs['long_name']='variance of column integrated MSE'
    varhsave.attrs['_FillValue']=-9999
    varhsave.attrs['GridType']='Gaussian Grid'    
    
    hSWsave.attrs['units']='J^2 m^-4 s^-1'
    hSWsave.attrs['long_name']= 'product of hanom and SWanom'
    hSWsave.attrs['_FillValue']=-9999
    hSWsave.attrs['GridType']='Gaussian Grid'    
    
    hLWsave.attrs['units']='J^2 m^-4 s^-1'
    hLWsave.attrs['long_name']='product of hanom and LWanom'
    hLWsave.attrs['_FillValue']=-9999
    hLWsave.attrs['GridType']='Gaussian Grid'   
    
    hSEFsave.attrs['units']='J^2 m^-4 s^-1'
    hSEFsave.attrs['long_name']= 'product of hanom and SEFanom'
    hSEFsave.attrs['_FillValue']=-9999
    hSEFsave.attrs['GridType']='Gaussian Grid'
    
    windsave.attrs['units']='m/s'
    windsave.attrs['long_name']='maximum wind speed'
    windsave.attrs['_FillValue']=-9999
    windsave.attrs['GridType']='Gaussian Grid'
    
    boxlat.attrs['units']='Degrees'
    boxlat.attrs['long_name']='Latitude'
    boxlat.attrs['_FillValue']=-9999
    boxlat.attrs['GridType']='Gaussian Latitude Spacing'
   
    boxlon.attrs['units']='Degrees'
    boxlon.attrs['long_name']='Longitude'
    boxlon.attrs['_FillValue']=-9999
    boxlon.attrs['GridType']='Gaussian Longitude Spacing'
    
    centerlatsave.attrs['units']='Degrees'
    centerlatsave.attrs['long_name']='Center latitude of storm'
    centerlatsave.attrs['_FillValue']=-9999
    centerlatsave.attrs['GridType']='Gaussian Latitude Spacing'
   
    centerlonsave.attrs['units']='Degrees'
    centerlonsave.attrs['long_name']='Center longitude of storm'
    centerlonsave.attrs['_FillValue']=-9999
    centerlonsave.attrs['GridType']='Gaussian Longitude Spacing'
    
    varbudget_ds = xr.Dataset({'SW':SWsave, 'LW':LWsave, 'SEF':SEFsave, 'wind':windsave, 'latitude':boxlat, \
            'longitude':boxlon, 'centerlat':centerlatsave, 'centerlon':centerlonsave, 'year':yearsave, \
            'month':monthsave, 'day':daysave, 'hour':hoursave, 'h':hsave, 'varh':varhsave, 'hSW':hSWsave, \
            'hLW':hLWsave, 'hSEF':hSEFsave}) 
    varbudget_ds.to_netcdf('/home/cdirkes/eraint/varbudget_output/new_varbudget.eraiv2.'+str(current_year)+'.nc', 'w', format='NETCDF4')
    print('\n')

