"""
This is the new way of selecting latbox and lonbox of the CFSR 
reanalysis data set. Selected by finding the index of the TempestTrack lat and 
lon points and selecting a box that is a certain number of grid points
below and above the center in either direction for a latbox and lonbox
that is approximately 10 degrees.
"""
import numpy as np 
import pandas as pd
import xarray as xr
import sys

# define the filepath of where to find the tracks
tempest_filepath = r'/home/awing/ibtracs/trajectories.txt.CFSR'

def boxavg(thing,lat,lon):
    """
    Calculate cosine(latitude)-weighted box average 
    e.g., h_avg=boxavg(h,lat)
    Important: h and lat must already be restricted to box of interest
    """
    
#       Define weights, *(idata*0+1) magically repeats it to
#       match dimensions of idata
    # wgt = np.cos(np.deg2rad(lat))*(thing*0+1)
    # thing_avg = (wgt*thing).sum()/wgt.sum() #sum over all grid points in box
    coslat_values=np.transpose(np.tile(np.cos(np.deg2rad(lat)),(len(lon),1)))
    thing1 = thing*coslat_values
    thing2 = thing1/thing1
    average=np.nansum(np.nansum(thing1,0))/np.nansum(np.nansum(coslat_values*thing2,0))

    return average

start_year = 1980 # beginning of most TempestExtremes tracks is 1980
end_year = 2016 # depends on how far the dataset goes (2018)
#----------Get the data organized to ultimately group by storm-----------#


df = pd.read_csv(tempest_filepath,sep='\t',header=None,names=\
                 ['flag','num_timesteps','dummy','lon','lat','pressure (hPa)',\
                  'windspeed (m/s)','dummy2','year','month','day','hour'])
# drop columns we don't need
df = df.drop(columns=(['dummy','dummy2'])) 
# add in a flag so that it knows when to 
df_starts = df[df['flag']=='start']
# start with assigning a stormID of 1 for the dataset 
storm_id = 1
# loop through each storm at a time and add in the appropriate storm_ids
for idx, num_steps in zip(df_starts.index, df_starts['num_timesteps'].values):
    # add in a column for the storm ID, arbitrary for each dataset
    df.loc[idx:idx+num_steps+1,'stormid'] = storm_id 
    # add 1 to the stormID each time you get to the end of a particular storm\
    # track to continue the loop
    storm_id += 1   

# identify and remove the flag column and num_timesteps column since we've \
# already added in the stormIDS
df = df[df['flag']!='start']
df = df.drop(columns=(['flag','num_timesteps'])).reset_index(drop=True)

# do some formatting to make some of the column headings formatted the way I want them
df.loc[:,'pressure (hPa)'] = df.loc[:,'pressure (hPa)'] / 100.
df.loc[:,'year'] = df.loc[:,'year'].astype(int).astype(str)
df.loc[:,'month'] = df.loc[:,'month'].astype(int).astype(str)
df.loc[:,'day'] = df.loc[:,'day'].astype(int).astype(str)
df.loc[:,'hour'] = df.loc[:,'hour'].astype(int).astype(str)
df.loc[:,'windspeed (m/s)'] = df.loc[:,'windspeed (m/s)']
df.loc[:,'hour'] = np.where(df['hour'].astype(int)<10,'0'+ df['hour'], df['hour'])

# create a timestamp for each time step so that it is easier to read in later
df.loc[:,'datetime'] = pd.to_datetime(df['year'] + ' ' + df['month'] + ' ' + df['day'] + ' ' + df['hour'])

#------------Now create a list of storm ID's by year for looping through\
#    and matching to the data----------#

ibtracs_by_year = {year: [] for year in range(start_year,end_year+1)} # make \
#an empty array of storm tracks by year, where you add 2 just in case

for storm in range(1,2586): # range of storms to loop through to generate \
#    list of storms by year. Note I manually found the last stormID (2372 1980-2015, 2586 1980-2018)
    sys.stdout.write(f'\r{storm}/2586')
    # show the progress in the command line
    
    # get the list of characteristics for the storm ID you're currently on
    ds_storm = df[df['stormid']==storm]
    # get the unique list of times for that storm
    times = ds_storm['datetime'].values
    # do some conversion to make it easier to work with
    times = pd.to_datetime(times) # convert pandas time to datetimeindex
    times = times.to_pydatetime() # convert to datetime.datetime objects
    # only include times for which the dataset is valid
    if (times[0].year < start_year and times[-1].year < start_year) or (\
    times[0].year > end_year and times[-1].year > end_year): 
        continue
    second_part_flag = False
    if times[0].year != times[-1].year: # this is to check if a storm starts in one year and ends in another, testing the first and last time in the storm
        ibtracs_by_year[times[-1].year].append(storm) # append the extra storm to the list to ensure all times are included in the year
    ibtracs_by_year[times[0].year].append(storm) # append the list of storms to the correct year, including ones that have parts of two years

# make a list to loop through each year--not the most efficient but I did this a long time ago haha
year_list=[] # make a list of years
for key in ibtracs_by_year:
    year_list.append(key)

"""-----------------------------------------------------------------------------------
This is where we start our main loops to extract all the data ------------------------
-----------------------------------------------------------------------------------"""
print('\n')
for current_year in range(2016,2017): #range (1980,2016) is 1980 through the end of 2015

    storm_times=[] # create an empty list to append to of times for all storms in the current year
    #outside the storm loop but inside the year loop rewrite the SEF as a function of lat and lon box
    
    #define the NaN arrays for various variable anomalies. don't need averages. also and boxlon
    # save both the variable and the variable anomaly, write 2 files, one w regular vars and one w anomalies
    #will need len(ib(current))
    
    #nlat and nlon will be whatever grid spacing makes a 10x10 degree box
    #for CFSR, lat and lon both have 0.5 degree grid spacing
    nlat = 21
    nlon = 21
    
    # nstorms = number of storms in that year --> in this case it is 41
    nstorms = len(ibtracs_by_year[current_year])
    # print(nstorms)
    
    #ntracks is maximum possible number of "observations" along the track of a storm
    #in this case the max num_timesteps is 86
    ntracks = max(df_starts['num_timesteps'])   
    # print(ntracks)
    
    #We need to make an empty 4D array of NaNs that have the dimensions lat x lon x nstorms x ntracks
    #want to store the regular variable and the box anomaly of net SW, net LW, and , and all those with hanom
    SWanomsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SWanomsave[:] = np.nan
    LWanomsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    LWanomsave[:] = np.nan
    SEFanomsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SEFanomsave[:] = np.nan
    SWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SWsave[:] = np.nan
    LWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    LWsave[:] = np.nan
    SEFsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    SEFsave[:] = np.nan
    lhfsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    lhfsave[:] = np.nan
    shfsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    shfsave[:] = np.nan
    
    boxlat = np.empty((nlat,nstorms,ntracks))
    boxlon = np.empty((nlon,nstorms,ntracks))
    boxlat[:] = np.nan
    boxlon[:] = np.nan
    
    yearsave = np.empty((nstorms, ntracks))
    yearsave[:] = np.nan   
    monthsave = np.empty((nstorms, ntracks))
    monthsave[:] = np.nan  
    daysave = np.empty((nstorms, ntracks))
    daysave[:] = np.nan    
    hoursave = np.empty((nstorms, ntracks))
    hoursave[:] = np.nan
    
    windsave = np.empty((nstorms, ntracks))
    windsave[:] = np.nan 
    
    hsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hsave[:] = np.nan
    
    varhsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    varhsave[:] = np.nan
    
    hSWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hSWsave[:] = np.nan
    hLWsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hLWsave[:] = np.nan   
    hSEFsave = np.empty((nlat,nlon,nstorms,ntracks)) 
    hSEFsave[:] = np.nan
    
    centerlatsave = np.empty((nstorms,ntracks))
    centerlatsave[:] = np.nan
    centerlonsave = np.empty((nstorms,ntracks))
    centerlonsave[:] = np.nan
    
    
    for s, storm in enumerate(ibtracs_by_year[current_year]): # start looping through every storm 
        # print(s,storm)
        # print('current year'+str(current_year))
        sys.stdout.write(f'\r{s}/{nstorms}')
        # generate a list of times valid for the whole year

        # this is a repeat of above where you're just formatting the times and making them how you want to be
        ds_storm = df[df['stormid']==storm]
        time_list = [tstamp.to_pydatetime() for tstamp in pd.to_datetime(ds_storm['datetime'].values)] # make a timelist and convert to datetime.datetime
        time_list = [t.replace(second=0, microsecond=0) for t in time_list]   # Round to nearest minute to avoid weird rounding errors
        storm_times.append(time_list) # append the time_list for each storm to the overall storm_times list altogether

        # Import ibtracs variables
        track_lat = ds_storm['lat'].values
        track_lon = ds_storm['lon'].values
        sid = ds_storm['stormid'].values # import storm ID values from USA dataset
        wind = ds_storm['windspeed (m/s)'].values # import wind speeds from USA dataset
        mslp = ds_storm['pressure (hPa)'].values # import mean sea level pressure from USA dataset in mb

        #################################################################################################

        # Loop through each time and assign the flux variables to the TC
        
        #################################################################################################
     
        time_datasets=[] # empty array of datasets by time        
        used_times = [] # list of times used

        for t, time in enumerate(time_list):
       
            if time_list[t].year != current_year: # don't accumulate precip for time steps in years other than current one
#*** might not need
                continue
            #doesn't matter which file we choose bc they all have the same lat/lon spacing 
            dslat = xr.open_dataset(r'/huracan/tank2/columbia/reanalysis/cfsr/2D/flux/1980/pgbh01.gdas.19800101-19800105.grb2.nc')
            latvar = np.array(dslat.lat)
            lonvar = np.array(dslat.lon)

            idxlat = pd.Index(latvar)
            idxlon = pd.Index(lonvar)
            latlist = idxlat.tolist()
            lonlist = idxlon.tolist()
            
            searchlat = dslat.lat.sel(lat = track_lat[t], method = 'nearest')
            ilat = latlist.index(searchlat)
        
            searchlon = dslat.lon.sel(lon = track_lon[t], method = 'nearest')
            ilon = lonlist.index(searchlon)

            ilat_up = ilat + 11
            ilat_down = ilat - 10
            ilon_up = ilon + 11
            ilon_down = ilon - 10
            
            max_lon = int(np.size(lonvar)-1)
            
            latbox = np.array(dslat.lat.isel(lat = slice(ilat_down,ilat_up)))            
            lonbox = np.array(dslat.lon.isel(lon = slice(ilon_down,ilon_up)))

            centerlat = track_lat[t]
            centerlon = track_lon[t]
            
            # get the formats of the dates and stuff the way you want them
            ym = time.strftime('%Y%m') # convert the year month time from datetime.datetime to a string
            ym = str(ym) # make the ym a string to be able to compare to the file name
            year = time.strftime('%Y') # convert the year from datetime.datetime to a string
            month = time.strftime('%m') # convert the month to a string
            month = month.zfill(2) #this adds a floating zero in front of the month
            day = time.strftime('%d')
            hour = time.strftime('%H') # convert the hour to a string
            m_minus1 = int(month)-1
            m_minus1 = str(m_minus1)
            m_minus1 = m_minus1.zfill(2) #this adds a floating zero in front of the month if it is one digit
            y_minus1 = int(year)-1
            y_minus1 = str(y_minus1)
            d_minus1 = int(day)-1
            d_minus1 = str(d_minus1)
            d_minus1 = d_minus1.zfill(2) #adds a floating zero if one digit
            MM = month.zfill(2)
            HH = hour.zfill(2)
            DD = day.zfill(2)
            YYYY = year
            MM_1 = m_minus1
            DD_1 = d_minus1
            YYYY_1 = y_minus1
     

            filepath = r'/huracan/tank2/columbia/reanalysis/cfsr/2D/flux/' #contains latent & sensible sfc fluxes
# ------------------------------------------------------------------------------------------------ 
# conditional statements to read in the variables depending on which day, month, and time it is.
#       BEFORE APRIL 2011:            
# option 1: day falls in the first 1-5 date range, but it is not 00z on the 1st 
# option 2: day falls in the second 6-10 date range, but it is not 00z on the 6th 
# option 3: day falls in the third 11-15 date range, but it is not 00z on the 11th
# option 4: day falls in the fourth 16-20 date range, but it is not 00z on the 16th
# option 5: day falls in the fifth 21-25 date range, but it is not 00z on the 21st 
# option 6: day falls in the last 26-28 date range February non leap year, but it is not 00z on the 26th
# option 7: day falls in the last 26-29 date range February leap year, but it is not 00z on the 26th 
# option 8: day falls in the last 26-30 date range of a month with 30 days, but it is not 00z on the 26th 
# option 9: day falls in the last 26-31 date range of a month with 31 days, but it is not 00z on the 26th 
# option 10: 00Z on the 6th
# option 11: 00Z on the 11th
# option 12: 00Z on the 16th
# option 13: 00Z on the 21st 
# option 14: 00Z on the 26th of February on NON Leap years
# option 15: 00Z on the 26th of February on Leap years
# option 16: 00Z on the 26th of a month with 30 days
# option 17: 00Z on the 26th of a month with 31 days
# option 18: 00Z on the 1st of March in a NON Leap Year 
# option 19: 00Z March 1st in a LEAP year 
# option 20: 00Z on the 1st of a month where the previous month has 30 days 
# option 21: 00Z on the 1st of a month where the previous month has 31 days (not January)
# option 22/ELSE: 00Z January 1st any year (this is fine for 1980 because we do actually have 1979)       
#       AFTER APRIL 2011:
# option 1: 00z April 1st 2011   
# option 2: 00z NOT the 1st day of the month  
# option 3: 00z 1st day of March non leap year
# option 4: 00z 1st day of March leap year
# option 5: 00z 1st day of month where previous month has 30 days
# option 6: 00z 1st day of month where previous month has 31 days
# option 7: 00z January 1st
# option 8: 6z any day
# option 9: 12z any day
# option 10 (else): 18z any day
# ------------------------------------------------------------------------------------------------        
            if time_list[t].year < 2011 or (time_list[t].year == 2011 and time_list[t].month < 4):  
    # Option 1            
    # day falls in the first 1-5 date range, but it is not 00z on the 1st            
                if (1 <= time_list[t].day <= 5) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 1)):
                    filename1 = '%s/pgbh01.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 1
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #Option 2    
    #if the day lies in the second range but is not 00z on the 6th 
    #normal condition for these ranges of days (6, 12, 18z, 00z NOT the 6th)
                elif (6 <= time_list[t].day <= 10) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 6)):
                    filename1 = '%s/pgbh01.gdas.%s%s06-%s%s10.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s06-%s%s10.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s06-%s%s10.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)    
                    
                    option = 2
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 3
    #if the day lies in the third range but is not 00z on the 11th 
    #normal condition for these ranges of days (6, 12, 18z, 00z NOT the 11th)
                elif (11 <= time_list[t].day <= 15) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 11)):
                    filename1 = '%s/pgbh01.gdas.%s%s11-%s%s15.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s11-%s%s15.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s11-%s%s15.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    option = 3
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
    #option 4
    #if the day lies in the fourth range but is not 00z on the 16th 
    #normal condition for these ranges of days (6, 12, 18z, 00z NOT the 16th)
                elif (16 <= time_list[t].day <= 20) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 16)):
                    filename1 = '%s/pgbh01.gdas.%s%s16-%s%s20.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s16-%s%s20.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s16-%s%s20.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 4
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 5
    #if the day lies in the fifth range but is not 00z on the 21st 
    #normal condition for these ranges of days (6, 12, 18z, 00z NOT the 21st)
                elif (21 <= time_list[t].day <= 25) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 21)):
                    filename1 = '%s/pgbh01.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 5
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 6
    #if the day lies in the last range but is not 00z on the 26th 
    #four different options depending on the number of days in the month
    #normal condition for these ranges of days (6, 12, 18z, 00z NOT the 26th)
    #first one: february 26-28 non leap year not 00z on the 26th
                elif (26 <= time_list[t].day <= 28) and time_list[t].month == 2 and \
                    (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 26)) and (time_list[t].year % 4 != 0): 
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s28.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s26-%s%s28.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s26-%s%s28.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 6
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 7
    #next one: february 26-29 leap year not 00z on the 26th
                elif (26 <= time_list[t].day <= 29) and time_list[t].month == 2 and \
                    (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 26)) and (time_list[t].year % 4 == 0): 
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s29.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s26-%s%s29.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s26-%s%s29.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 7
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 8
    #next one: day 26-30 month has 30 days not 00z on the 26th
                elif (26 <= time_list[t].day <= 30) and (time_list[t].month == 4 or \
                    time_list[t].month == 6 or time_list[t].month == 9 or \
                    time_list[t].month == 11) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 26)): 
                    
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s30.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s26-%s%s30.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s26-%s%s30.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 8
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 9
    #next one: day 26-31 month has 31 days not 00z on the 26th
                elif (26 <= time_list[t].day <= 31) and (time_list[t].month == 1 or \
                    time_list[t].month == 3 or time_list[t].month == 5 or \
                    time_list[t].month == 7 or time_list[t].month == 8 or time_list[t].month == 10 or \
                    time_list[t].month == 12) and (time_list[t].hour != 0 or (time_list[t].hour == 0 and time_list[t].day != 26)):
                	
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s31.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s26-%s%s31.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s26-%s%s31.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    
                    option = 9
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 10
    #if 00Z on the 6th, the pgbh05 and pgbh06 have to be from the previous range of days, 01-05
                elif time_list[t].day == 6 and time_list[t].hour == 0:
                    filename1 = '%s/pgbh01.gdas.%s%s06-%s%s10.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 10
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
    #option 11
    #if 00Z on the 11th, the pgbh05 and pgbh06 have to be from the previous range of days, 06-10
                elif time_list[t].day == 11 and time_list[t].hour == 0:
                    filename1 = '%s/pgbh01.gdas.%s%s11-%s%s15.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s06-%s%s10.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s06-%s%s10.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 11
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
    #option 12
    #if 00Z on the 16th, the pgbh05 and pgbh06 have to be from the previous range of days, 11-15
                elif time_list[t].day == 16 and time_list[t].hour == 0:
                    filename1 = '%s/pgbh01.gdas.%s%s16-%s%s20.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s11-%s%s15.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s11-%s%s15.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 12
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                                    
    #option 13
    #if 00Z on the 21st, the pgbh05 and pgbh06 have to be from the previous range of days, 16-20
                elif time_list[t].day == 21 and time_list[t].hour == 0:
                    filename1 = '%s/pgbh01.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s16-%s%s20.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s16-%s%s20.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 13
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
    #option 14
    #if 00Z on the 26th, the pgbh05 and pgbh06 have to be from the previous range of days, 21-25 AND it is FEBRUARY on a NON leap year 
                elif time_list[t].day == 26 and time_list[t].hour == 0 and \
                    time_list[t].month == 2 and (time_list[t].year % 4 != 0):
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s28.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 14
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 15
    #if 00Z on the 26th, the pgbh05 and pgbh06 have to be from the previous range of days, 21-25 AND it is FEBRUARY on a LEAP year 
                elif time_list[t].day == 26 and time_list[t].hour == 0 and \
                    time_list[t].month == 2 and (time_list[t].year % 4 == 0):
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s29.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 15
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
    #option 16
    #if 00Z on the 26th, the pgbh05 and pgbh06 have to be from the previous range of days, 21-25 AND the month has 30 days (sep, april, jun, nov)
                elif time_list[t].day == 26 and time_list[t].hour == 0 and \
                    (time_list[t].month == 4 or time_list[t].month == 6 or \
                     time_list[t].month == 9 or time_list[t].month == 11):
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s30.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 16
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 17
    #if 00Z on the 26th, the pgbh05 and pgbh06 have to be from the previous range of days, 21-25 AND the month has 31 days (jan, mar, may, jul, aug, oct, dec)
                elif time_list[t].day == 26 and time_list[t].hour == 0 and \
                    (time_list[t].month == 1 or time_list[t].month == 3 or \
                     time_list[t].month == 5 or time_list[t].month == 7 or \
                     time_list[t].month == 8 or time_list[t].month == 10 or \
                     time_list[t].month == 12):
                    filename1 = '%s/pgbh01.gdas.%s%s26-%s%s31.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename6 = '%s/pgbh06.gdas.%s%s21-%s%s25.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 17
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 18
    #if it is 00Z on the 1st of the month AND March in NON Leap year 
                elif time_list[t].day == 1 and time_list[t].hour == 0 and \
                    time_list[t].month == 3 and (time_list[t].year % 4 != 0):
                    filename1 = '%s/pgbh01.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s0226-%s0228.grb2.nc' %(YYYY,YYYY,YYYY)
                    filename6 = '%s/pgbh06.gdas.%s0226-%s0228.grb2.nc' %(YYYY,YYYY,YYYY)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 18
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 19
    #if it is 00Z on the 1st of the month AND March in a LEAP year 
                elif time_list[t].day == 1 and time_list[t].hour == 0 and \
                    time_list[t].month == 3 and (time_list[t].year % 4 == 0):
                    filename1 = '%s/pgbh01.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s0226-%s0229.grb2.nc' %(YYYY,YYYY,YYYY)
                    filename6 = '%s/pgbh06.gdas.%s0226-%s0229.grb2.nc' %(YYYY,YYYY,YYYY)
                    
                    ds1 = xr.open_dataset(filepath + filename1)        
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 19
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    
    #option 20
    #if 00z on the 1st where the previous month has 30 days
                elif (time_list[t].month == 5 or time_list[t].month == 7 or \
                      time_list[t].month == 10 or time_list[t].month == 12) and \
                      time_list[t].day == 1 and time_list[t].hour == 0:
                    filename1 = '%s/pgbh01.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s26-%s%s30.grb2.nc' %(YYYY,YYYY,MM_1,YYYY,MM_1)
                    filename6 = '%s/pgbh06.gdas.%s%s26-%s%s30.grb2.nc' %(YYYY,YYYY,MM_1,YYYY,MM_1)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 20
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
         
    #option 21
    #if 00z on the 1st where the previous month has 31 days 
                elif (time_list[t].month == 2 or time_list[t].month == 4 or time_list[t].month == 6 or \
                      time_list[t].month == 8 or time_list[t].month == 9 or time_list[t].month == 11) and\
                      time_list[t].day == 1 and time_list[t].hour == 0:
                    filename1 = '%s/pgbh01.gdas.%s%s01-%s%s05.grb2.nc' %(YYYY,YYYY,MM,YYYY,MM)
                    filename5 = '%s/pgbh05.gdas.%s%s26-%s%s31.grb2.nc' %(YYYY,YYYY,MM_1,YYYY,MM_1)
                    filename6 = '%s/pgbh06.gdas.%s%s26-%s%s31.grb2.nc' %(YYYY,YYYY,MM_1,YYYY,MM_1)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    option = 21
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    #option 22
    #if it is 00Z on the 1st of the month IN January 
    #elif time_list[t].day == 1 and time_list[t].hour == 0 and time_list[t].month == 1:
                else:
    
                    filename1 =  '%s/pgbh01.gdas.%s0101-%s0105.grb2.nc' %(YYYY,YYYY,YYYY)
                    filename5 = '%s/pgbh05.gdas.%s1226-%s1231.grb2.nc' %(YYYY_1,YYYY_1,YYYY_1)
                    filename6 = '%s/pgbh06.gdas.%s1226-%s1231.grb2.nc' %(YYYY_1,YYYY_1,YYYY_1)
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)   
                    
                    option = 22
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
    
    
                if time_list[t].day == 1 or time_list[t].day == 6 or time_list[t].day == 11 \
                    or time_list[t].day == 16 or time_list[t].day == 21 or time_list[t].day == 26: 
                        if time_list[t].hour == 0: 
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2                        
                            
                        elif time_list[t].hour == 6:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
    
                        elif time_list[t].hour == 12:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 18:    
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 2, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                        
                elif time_list[t].day == 2 or time_list[t].day == 7 or time_list[t].day == 12 \
                    or time_list[t].day == 17 or time_list[t].day == 22 or time_list[t].day == 27:
                        if time_list[t].hour == 0:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 3, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 6:    
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 12:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 18:     
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2        
                        
                elif time_list[t].day == 3 or time_list[t].day == 8 or time_list[t].day == 13 \
                    or time_list[t].day == 18 or time_list[t].day == 23 or time_list[t].day == 28:   
                        if time_list[t].hour == 0:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 7, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 6:   
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 8, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 12:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 9, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 18:  
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 10, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2                            
    
                elif time_list[t].day == 4 or time_list[t].day == 9 or time_list[t].day == 14 \
                    or time_list[t].day == 19 or time_list[t].day == 24 or time_list[t].day == 29:
                        if time_list[t].hour == 0:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 11, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 6:    
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 12, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 12:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 18:  
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                        
                elif time_list[t].day == 5 or time_list[t].day == 10 or time_list[t].day == 15 \
                    or time_list[t].day == 20 or time_list[t].day == 25 or time_list[t].day == 30:
                        if time_list[t].hour == 0:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 6:   
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 16, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 12:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 17, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 18:  
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 18, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                else: #day 31 is all that's left
                        # print('option: '+str(option))
                        if time_list[t].hour == 0:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 19, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 6:    
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 20, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 12:
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 21, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
                        elif time_list[t].hour == 18:  
                            lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                            
                            shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf5 = ds5.SHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf6 = ds6.SHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            shf = ((shf6*6)-(shf5*5)+shf1)/2
                            
                            sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup5 = ds5.USWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup6 = ds6.USWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                            
                            sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                            
                            tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup5 = ds5.USWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup6 = ds6.USWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                            
                            tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                            
                            slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                            
                            slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                            
                            tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                            tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                            
            else: # after april 2011
                #option 1: 00z April 1st 2011
                if time_list[t].hour == 0 and time_list[t].day == 1 and time_list[t].month == 4 and time_list[t].year == 2011:
                    option = 1
                    filename1 = '2011/cdas1.20110401.pgrbh.grb2.nc'
                    filename5 = '2011/pgbh05.gdas.20110326-20110331.grb2.nc'
                    filename6 = '2011/pgbh06.gdas.20110326-20110331.grb2.nc'
                    
                    ds1 = xr.open_dataset(filepath + filename1)
                    ds5 = xr.open_dataset(filepath + filename5)
                    ds6 = xr.open_dataset(filepath + filename6)
                    
                    lhf1 = ds1.LHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds5.LHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds6.LHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds1.SHTFL_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds5.SHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds6.SHTFL_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds1.USWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds5.USWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds6.USWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds1.DSWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds5.DSWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds6.DSWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds1.USWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds5.USWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds6.USWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds1.DSWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds5.DSWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds6.DSWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds1.ULWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds5.ULWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds6.ULWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds1.DLWRF_L1_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds5.DLWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds6.DLWRF_L1_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds1.ULWRF_L8_Avg_1.isel(time = 0, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds5.ULWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds6.ULWRF_L8_Avg_1.isel(time = -1, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2   
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                
                #option 2: 00z NOT the 1st day of the month
                elif time_list[t].hour == 0 and time_list[t].day != 1:
                    option = 2
                    if time_list[t].hour == 0 and time_list[t].day == 10 and time_list[t].month == 1 and time_list[t].year == 2012:
                                    #this random file is missing in the dataset
                        lhf = np.empty((nlat,nlon)) 
                        shf = np.empty((nlat,nlon)) 
                        sswup = np.empty((nlat,nlon)) 
                        sswdown = np.empty((nlat,nlon)) 
                        tswup = np.empty((nlat,nlon)) 
                        tswdown = np.empty((nlat,nlon)) 
                        slwup = np.empty((nlat,nlon)) 
                        slwdown = np.empty((nlat,nlon)) 
                        tlw = np.empty((nlat,nlon)) 
                        #set them to NaN
                        lhf[:] = np.nan
                        shf[:] = np.nan
                        sswup[:] = np.nan
                        sswdown[:] = np.nan
                        tswup[:] = np.nan
                        tswdown[:] = np.nan
                        slwup[:] = np.nan
                        slwdown[:] = np.nan
                        tlw[:] = np.nan
                    else:
                        filename = '%s/cdas1.%s%s%s.pgrbh.grb2.nc' %(YYYY,YYYY,MM,DD_1)
                        ds = xr.open_dataset(filepath + filename)
                        
                        lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                        
                        shf1 = ds.SHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        shf5 = ds.SHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        shf6 = ds.SHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        shf = ((shf6*6)-(shf5*5)+shf1)/2
                        
                        sswup1 = ds.USWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        sswup5 = ds.USWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        sswup6 = ds.USWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                        
                        sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                        
                        tswup1 = ds.USWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tswup5 = ds.USWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tswup6 = ds.USWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                        
                        tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                        
                        slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                        
                        slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                        
                        tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                        tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                        
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                
                #option 3: 00z 1st day of March non leap year
                elif time_list[t].hour == 0 and time_list[t].day == 1 and \
                    time_list[t].month == 3 and (time_list[t].year % 4 != 0):
                    option = 3
                    filename = '%s/cdas1.%s0228.pgrbh.grb2.nc' %(YYYY,YYYY)
                    ds = xr.open_dataset(filepath + filename)
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
             
                #option 4: 00z 1st day of March LEAP year
                elif time_list[t].hour == 0 and time_list[t].day == 1 and \
                    time_list[t].month == 3 and (time_list[t].year % 4 == 0):
                    option = 4
                    filename = '%s/cdas1.%s0229.pgrbh.grb2.nc' %(YYYY,YYYY)
                    ds = xr.open_dataset(filepath + filename)  
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2   
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
                #option 5: 00Z AND 1st of the month AND (months 5 OR 7 OR 10 OR 12), months where the previous month has 30 days
                elif (time_list[t].month == 5 or time_list[t].month == 7 or time_list[t].month == 10 or \
                      time_list[t].month == 12) and time_list[t].day == 1 and time_list[t].hour == 0:
                    option = 5
                    filename = '%s/cdas1.%s%s30.pgrbh.grb2.nc' %(YYYY,YYYY,MM_1)
                    ds = xr.open_dataset(filepath + filename)  
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
                #option 6: 00Z AND 1st of the month AND (months 2 OR 4 OR 6 OR 8 OR 9 OR 11), months where the previous month has 31 days
                elif (time_list[t].month == 2 or time_list[t].month == 4 or time_list[t].month == 6 or \
                      time_list[t].month == 8 or time_list[t].month == 9 or time_list[t].month == 11) and\
                      time_list[t].day == 1 and time_list[t].hour == 0:
                    option = 6
                    filename = '%s/cdas1.%s%s31.pgrbh.grb2.nc' %(YYYY,YYYY,MM_1)
                    ds = xr.open_dataset(filepath + filename) 
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
                #option 7: 00Z on january 1st
                elif time_list[t].hour == 0 and time_list[t].day == 1 and time_list[t].month == 1:
                    option = 7
                    filename = '%s/cdas1.%s1231.pgrbh.grb2.nc' %(YYYY_1,YYYY_1)
                    ds = xr.open_dataset(filepath + filename) 
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 33, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 31, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 32, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                
                #option 8: 6z any day
                elif time_list[t].hour == 6:
                    option = 8
                    filename = '%s/cdas1.%s%s%s.pgrbh.grb2.nc' %(YYYY,YYYY,MM,DD)
                    ds = xr.open_dataset(filepath + filename) 
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 6, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 4, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 5, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
                #option 9: 12z any day
                elif time_list[t].hour == 12:  
                    option = 9
                    filename = '%s/cdas1.%s%s%s.pgrbh.grb2.nc' %(YYYY,YYYY,MM,DD)
                    ds = xr.open_dataset(filepath + filename) 
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 15, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 13, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 14, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                        
                #option 10: 18z any day
                else: # time_list[t].hour == 18:
                    option = 10
                    filename = '%s/cdas1.%s%s%s.pgrbh.grb2.nc' %(YYYY,YYYY,MM,DD)
                    ds = xr.open_dataset(filepath + filename) 
                    
                    lhf1 = ds.LHTFL_L1_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf5 = ds.LHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf6 = ds.LHTFL_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    lhf = ((lhf6*6)-(lhf5*5)+lhf1)/2
                    
                    shf1 = ds.SHTFL_L1_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf5 = ds.SHTFL_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf6 = ds.SHTFL_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    shf = ((shf6*6)-(shf5*5)+shf1)/2
                    
                    sswup1 = ds.USWRF_L1_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup5 = ds.USWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup6 = ds.USWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswup = ((sswup6*6)-(sswup5*5)+sswup1)/2
                    
                    sswdown1 = ds.DSWRF_L1_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown5 = ds.DSWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown6 = ds.DSWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    sswdown = ((sswdown6*6)-(sswdown5*5)+sswdown1)/2
                    
                    tswup1 = ds.USWRF_L8_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup5 = ds.USWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup6 = ds.USWRF_L8_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswup = ((tswup6*6)-(tswup5*5)+tswup1)/2
                    
                    tswdown1 = ds.DSWRF_L8_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown5 = ds.DSWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown6 = ds.DSWRF_L8_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tswdown = ((tswdown6*6)-(tswdown5*5)+tswdown1)/2
                    
                    slwup1 = ds.ULWRF_L1_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup5 = ds.ULWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup6 = ds.ULWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwup = ((slwup6*6)-(slwup5*5)+slwup1)/2
                    
                    slwdown1 = ds.DLWRF_L1_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown5 = ds.DLWRF_L1_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown6 = ds.DLWRF_L1_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    slwdown = ((slwdown6*6)-(slwdown5*5)+slwdown1)/2
                    
                    tlw1 = ds.ULWRF_L8_Avg_1.isel(time = 24, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw5 = ds.ULWRF_L8_Avg_1.isel(time = 22, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw6 = ds.ULWRF_L8_Avg_1.isel(time = 23, lat = slice(ilat_down,ilat_up), lon = slice(ilon_down,ilon_up))  
                    tlw = ((tlw6*6)-(tlw5*5)+tlw1)/2
                    
                    
                    # print('option: '+str(option)+' time: '+str(YYYY) + str(MM) + str(DD) + str(HH))
                    
                    
            ssw = sswdown - sswup
            tsw = tswdown - tswup 
            
            slw = slwup - slwdown 
            
            SW = tsw - ssw
            LW = slw - tlw
            SEF = lhf + shf   
            
#--------------------------------------------------------------------------------------#
#                                 THIS ENDS THE CONDITIONALS                           #
#--------------------------------------------------------------------------------------#
                
            #still inside innermost loop over storms but out of the conditonal
            #read in h from huracan 
            #now define anoms with the 4D indices    
                
            #there are a couple missing days, just set them to NaN
            if (time_list[t].year == 2013 and time_list[t].month == 5 and time_list[t].day == 15) \
            or (time_list[t].year == 2013 and time_list[t].month == 6 and 16 <= time_list[t].day <= 17)\
            or (time_list[t].year == 2016 and time_list[t].month == 3 and time_list[t].day == 29) \
            or (time_list[t].year == 2016 and time_list[t].month == 3 and time_list[t].day == 31) \
            or (time_list[t].year == 2016 and time_list[t].month == 4 and 1 <= time_list[t].day <= 2) \
            or (time_list[t].year == 2016 and time_list[t].month == 9 and time_list[t].day < 4) \
            or (time_list[t].year == 2016 and time_list[t].month == 12 and time_list[t].day == 17):
                h = np.empty((nlat,nlon))
                h[:] = np.nan
            else:
                hpath = '/huracan/tank2/columbia/reanalysis/cfsr/2D/h/'
                hfilename = 'cfsr.h.%s%s%s%s.nc' % (YYYY,MM,DD,HH)
                dsh = xr.open_dataset(hpath + hfilename)
                h = dsh.h.isel(latitude = slice(ilat_down,ilat_up), longitude = slice(ilon_down,ilon_up))

                        #landmask stuff
            dslm = xr.open_dataset(r'/home/cdirkes/cfsr/landmask/landmask.nc')
            lm = dslm.landmask.isel(time = 0)#, latitude = slice(ilat_down,ilat_up), longitude = slice(ilon_down,ilon_up))
            lonlm = dslm.lon #ead in longitude variable from that dataset
            latlm = dslm.lat #read in latitude variable from that dataset
            idxlatlm = pd.Index(latlm)
            idxlonlm = pd.Index(lonlm)
            latlistlm = idxlatlm.tolist()
            lonlistlm = idxlonlm.tolist()
            lsm = np.zeros((nlat,nlon))
            for a in range(0,len(latbox)):
                for o in range(0,len(lonbox)):
                    testlat = np.array(latbox[a])
                    testlon = np.array(lonbox[o])
                    
                    searchlatlm = np.array(dslm.lat.sel(lat = testlat, method = 'nearest'))
                    # print(a, testlat, np.array(searchlat),np.array(searchlatlm))
                    lmlat = latlistlm.index(searchlatlm)
                    
                    searchlonlm = np.array(dslm.lon.sel(lon = testlon, method = 'nearest'))
                    # print(o, testlon, np.array(searchlon),np.array(searchlonlm))
                    lmlon = lonlistlm.index(searchlonlm)
                    if lm[lmlat,lmlon] > 0.2:
                        lsm[a,o] = np.nan   
            
            lsm = np.array(lsm)
            # print(lsm)
            h = np.array(h)
            SEF = np.array(SEF)
            SW = np.array(SW)
            LW = np.array(LW)
            
            if (np.shape(lsm) != np.shape(SW)) or (np.shape(lsm) != np.shape(h)) or (np.shape(lsm) != np.shape(LW)) or (np.shape(lsm) != np.shape(SEF)):
                lsm = lsm[:,0:(np.size(lonbox))]              
            
            havg = boxavg(h, latbox, lonbox)
            SWavg = boxavg(SW, latbox, lonbox)
            LWavg = boxavg(LW, latbox, lonbox)
            SEFavg = boxavg(SEF, latbox, lonbox)    
            
            h = h + lsm
            SEF = SEF + lsm
            SW = SW + lsm
            LW = LW + lsm

            hanom = h - havg
            hsave[0:len(latbox),0:len(lonbox),s,t] = h
            varhsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(hanom))
                            
            SWanom = SW - SWavg
            SWanomsave[0:len(latbox),0:len(lonbox),s,t] = SW - SWavg
            
            LWanom = LW - LWavg
            LWanomsave[0:len(latbox),0:len(lonbox),s,t] = LW - LWavg
            
            SEFanom = SEF - SEFavg
            SEFanomsave[0:len(latbox),0:len(lonbox),s,t] = SEF - SEFavg
            
            windsave[s,t] = wind[t]

            boxlat[0:len(latbox),s,t] = latbox
            boxlon[0:len(lonbox), s, t] = lonbox
            
            centerlatsave[s,t] = track_lat[t]
            centerlonsave[s,t] = track_lon[t]
             #change len to latbox and longbox    
            SWsave[0:len(latbox),0:len(lonbox),s,t] = SW
            LWsave[0:len(latbox),0:len(lonbox),s,t] = LW
            SEFsave[0:len(latbox),0:len(lonbox),s,t] = SEF
            lhfsave[0:len(latbox),0:len(lonbox),s,t] = lhf
            shfsave[0:len(latbox),0:len(lonbox),s,t] = shf
            
            hSWsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(SWanom))
            hLWsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(LWanom))
            hSEFsave[0:len(latbox),0:len(lonbox),s,t] = np.multiply(np.array(hanom), np.array(SEFanom))
                     
            yearsave[s,t] = time_list[t].year
            monthsave[s,t] = time_list[t].month
            daysave[s,t] = time_list[t].day
            hoursave[s,t] = time_list[t].hour
                        
            #convert them all into xarrays
            SWsave = xr.core.dataarray.DataArray(SWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            LWsave = xr.core.dataarray.DataArray(LWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            SEFsave = xr.core.dataarray.DataArray(SEFsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            lhfsave = xr.core.dataarray.DataArray(lhfsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            shfsave = xr.core.dataarray.DataArray(shfsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])            
            
            SWanomsave = xr.core.dataarray.DataArray(SWanomsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            LWanomsave = xr.core.dataarray.DataArray(LWanomsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            SEFanomsave = xr.core.dataarray.DataArray(SEFanomsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            
            hsave = xr.core.dataarray.DataArray(hsave, dims=['nlat', 'nlon', 'nstorms', 'ntracks'])
            varhsave = xr.core.dataarray.DataArray(varhsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            hSWsave = xr.core.dataarray.DataArray(hSWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            hLWsave = xr.core.dataarray.DataArray(hLWsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])
            hSEFsave = xr.core.dataarray.DataArray(hSEFsave, dims=['nlat', 'nlon', 'nstorms','ntracks'])

            
            windsave = xr.core.dataarray.DataArray(windsave, dims=['nstorms', 'ntracks'])

            boxlat = xr.core.dataarray.DataArray(boxlat, dims=['nlat', 'nstorms','ntracks'])
            boxlon = xr.core.dataarray.DataArray(boxlon, dims=['nlon', 'nstorms','ntracks'])
            centerlatsave = xr.core.dataarray.DataArray(centerlatsave, dims=['nstorms','ntracks'])
            centerlonsave = xr.core.dataarray.DataArray(centerlonsave, dims=['nstorms','ntracks'])
            
            yearsave = xr.core.dataarray.DataArray(yearsave, dims=['nstorms', 'ntracks'])
            monthsave = xr.core.dataarray.DataArray(monthsave, dims=['nstorms', 'ntracks'])
            daysave = xr.core.dataarray.DataArray(daysave, dims=['nstorms', 'ntracks'])
            hoursave = xr.core.dataarray.DataArray(hoursave, dims=['nstorms', 'ntracks'])
            
    """
    Write variable (already defined as a data array) to a netcdf file. making sure that it
    is within the first nested loop over all the years but not within the storms or tracks
    Need: SEF, SW, LW, all their anomalies, and lat and lon
    """
    #save all the units and attributes of each variable
    yearsave.attrs['units'] = 'year of specific storm'
    yearsave.attrs['long_name'] = 'year'
    
    monthsave.attrs['units'] = 'month of specific storm'
    monthsave.attrs['long_name'] = 'month'
    
    daysave.attrs['units'] = 'day of specific storm'
    daysave.attrs['long_name'] = 'day'
    
    hoursave.attrs['units'] = '6hr accumulation centered around hour'
    hoursave.attrs['long_name'] = 'hour, UTC'

    SWsave.attrs['units']='W/m^2'
    SWsave.attrs['long_name']='Net shortwave flux'
    SWsave.attrs['_FillValue']=-9999
    SWsave.attrs['GridType']='Lat/lon grid'
    
    LWsave.attrs['units']='W/m^2'
    LWsave.attrs['long_name']='Net longwave flux'
    LWsave.attrs['_FillValue']=-9999
    LWsave.attrs['GridType']='Lat/lon Grid'
  
    SEFsave.attrs['units']='W/m^2'
    SEFsave.attrs['long_name']='Surface enthalpy flux'
    SEFsave.attrs['_FillValue']=-9999
    SEFsave.attrs['GridType']='Lat/lon Grid'
    
    hsave.attrs['units']='J/m^2'
    hsave.attrs['long_name']='column integrated frozen moist static energy'
    hsave.attrs['_FillValue']=-9999
    hsave.attrs['GridType']='Lat/lon Grid'
    
    varhsave.attrs['units']='J^2/m^4'
    varhsave.attrs['long_name']='variance of column integrated FMSE'
    varhsave.attrs['_FillValue']=-9999
    varhsave.attrs['GridType']='Lat/lon Grid'    
    
    hSWsave.attrs['units']='J^2 m^-4 s^-1'
    hSWsave.attrs['long_name']= 'product of hanom and SWanom'
    hSWsave.attrs['_FillValue']=-9999
    hSWsave.attrs['GridType']='Lat/lon Grid'    
    
    hLWsave.attrs['units']='J^2 m^-4 s^-1'
    hLWsave.attrs['long_name']='product of hanom and LWanom'
    hLWsave.attrs['_FillValue']=-9999
    hLWsave.attrs['GridType']='Lat/lon Grid'   
    
    hSEFsave.attrs['units']='J^2 m^-4 s^-1'
    hSEFsave.attrs['long_name']= 'product of hanom and SEFanom'
    hSEFsave.attrs['_FillValue']=-9999
    hSEFsave.attrs['GridType']='Lat/lon Grid'
    
    windsave.attrs['units']='m/s'
    windsave.attrs['long_name']='maximum wind speed'
    windsave.attrs['_FillValue']=-9999
    windsave.attrs['GridType']='Lat/lon Grid'
    
    boxlat.attrs['units']='Degrees'
    boxlat.attrs['long_name']='Latitude'
    boxlat.attrs['_FillValue']=-9999
    boxlat.attrs['GridType']='0.5 deg Latitude Spacing'
   
    boxlon.attrs['units']='Degrees'
    boxlon.attrs['long_name']='Longitude'
    boxlon.attrs['_FillValue']=-9999
    boxlon.attrs['GridType']='0.625 deg Longitude Spacing'
    
    centerlatsave.attrs['units']='Degrees'
    centerlatsave.attrs['long_name']='Center latitude of storm'
    centerlatsave.attrs['_FillValue']=-9999
    centerlatsave.attrs['GridType']='0.5 deg Latitude Spacing'
   
    centerlonsave.attrs['units']='Degrees'
    centerlonsave.attrs['long_name']='Center longitude of storm'
    centerlonsave.attrs['_FillValue']=-9999
    centerlonsave.attrs['GridType']='0.625 Longitude Spacing'
    
    lhfsave.attrs['units']='W/m^2'
    lhfsave.attrs['long_name']='latent heat flux'
    lhfsave.attrs['_FillValue']=-9999
    lhfsave.attrs['GridType']='Lat/lon Grid'
  
    shfsave.attrs['units']='W/m^2'
    shfsave.attrs['long_name']='sensible heat flux'
    shfsave.attrs['_FillValue']=-9999
    shfsave.attrs['GridType']='Lat/lon Grid'
    
    varbudget_ds = xr.Dataset({'SW':SWsave, 'LW':LWsave, 'SEF':SEFsave, 'wind':windsave, 'latitude':boxlat, \
            'longitude':boxlon, 'centerlat':centerlatsave, 'centerlon':centerlonsave, 'year':yearsave, \
            'month':monthsave, 'day':daysave, 'hour':hoursave, 'h':hsave, 'varh':varhsave, 'hSW':hSWsave, \
            'hLW':hLWsave, 'hSEF':hSEFsave, 'lhf':lhfsave, 'shf':shfsave}) 
    #varbudget_ds.to_netcdf('/home/cdirkes/cfsr/varbudget_output/test_varbudget.cfsr.'+str(current_year)+'.nc', 'w', format='NETCDF4')
    varbudget_ds.to_netcdf('/home/awing/test_varbudget.cfsr.'+str(current_year)+'.nc', 'w', format='NETCDF4')
    print('\n')

