!===============================================================================
! SVN $Id: gen_domain.F90 6673 2007-09-28 22:11:15Z kauff $
! SVN $URL: https://svn-ccsm-models.cgd.ucar.edu/tools/mapping/gen_domain/trunk_tags/gen_domain_071001/gen_domain.F90 $
!===============================================================================

PROGRAM make_domain

   IMPLICIT NONE

   !--- includes ---
   include 'netcdf.inc'       ! netCDF defs

   !--- domain data ---
   integer         ::    n      ! size of 1d domain
   integer         ::   ni      ! size of i-axis of 2d domain
   integer         ::   nj      ! size of j-axis of 2d domain
   integer         ::   nv = 4  ! assume retalinear grid
   real*8 ,pointer ::   xc(  :) ! x-coords of center    
   real*8 ,pointer ::   yc(  :) ! y-coords of center   
   real*8 ,pointer ::   xv(:,:) ! x-coords of verticies 
   real*8 ,pointer ::   yv(:,:) ! y-coords of verticies 
   integer,pointer :: mask(  :) ! domain mask           
   real*8 ,pointer :: area(  :) ! cell area
   real*8 ,pointer :: frac(  :) ! cell fraction

   !--- for mapping ---
   logical         :: complf    ! flag for computing landfrac
   integer         :: ns        ! size of wgts list
   integer,pointer :: col (  :) ! column index
   integer,pointer :: row (  :) ! row index
   real*8 ,pointer :: S   (  :) ! wgts
   integer         :: na        ! size of source array
   integer,pointer :: maska( :) ! mask of source array, integer
   real*8 ,pointer :: fraca( :) ! mask of source array, real
   real*8          :: eps       ! allowable frac error
   real*8          :: frac_min  ! min frac value before being set to fminval
   real*8          :: frac_max  ! max frac value before being set to fmaxval
   real*8          :: fminval   ! set frac to zero if frac < fminval
   real*8          :: fmaxval   ! set frac to one  if frac > fmaxval

   !--- local ---
   character(LEN= 8)     :: cdate      ! wall clock date
   character(LEN=10)     :: ctime      ! wall clock time
   integer               :: rcode      ! routine return error code
   integer               :: fid        ! nc file     ID
   integer               :: vid        ! nc variable ID
   integer               :: did        ! nc dimension ID
   integer               :: vdid(3)    ! vector of nc dimension ID
   integer               :: i,j,k      ! generic indicies
   character,allocatable :: strarr(:)  ! variable length char string
   character(LEN=256)    :: str        ! fixed    length char string
   character(LEN=256)    :: str_title  ! global attribute str - title
   character(LEN=256)    :: str_source ! global attribute str - source
   character(LEN=256)    :: str_da     ! global attribute str - domain_a
   character(LEN=256)    :: str_db     ! global attribute str - domain_b
   character(LEN=256)    :: str_grido  ! global attribute str - grid_file_ocn
   character(LEN=256)    :: str_grida  ! global attribute str - grid_file_atm
   character(LEN=256)    :: usercomment! user comment from namelist
   character(LEN=256)    :: user       ! user name
   integer               :: strlen     ! (trimmed) length of string
   integer               :: nf         ! number of files counter
   character(LEN=256)    :: fn_in      ! file name ( input nc file)
   character(LEN=256)    :: fn_out     ! current output file name
   character(LEN=256)    :: fn1_out    ! file name (output nc file) for grid _a
   character(LEN=256)    :: fn2_out    ! file name (output nc file) for grid _b
   character(LEN=32)     :: attstr     ! netCDF attribute name string
   character(LEN=2)      :: suffix     ! suffix _a or _b sets input grid
   integer               :: set_fv_pole_yc ! fix pole ycs on this grid [0,1,2]
   logical               :: pole_fix   ! fix pole ycs
   real*8,parameter      :: pi  = 3.14159265358979323846
   real*8,parameter      :: c0  = 0.00000000000000000000
   real*8,parameter      :: c1  = 1.00000000000000000000
   real*8,parameter      :: c90 = 90.0000000000000000000
   character(*),parameter :: version = 'SVN $Id: gen_domain.F90 6673 2007-09-28 22:11:15Z kauff $'

   NAMELIST / in_parm / fn_in, fn1_out, fn2_out, usercomment, eps, fminval, fmaxval, set_fv_pole_yc

   !--- formats ---
!   character(LEN=*),parameter :: F00 = "(120a )"
!   character(LEN=*),parameter :: F02 = "(a,5i6,i12)"
!   character(LEN=*),parameter :: F10=&
!   & "('Data created: 'i4,'-',i2,2('-',i2),' ',i2,2('-',i2),' ')"

!-------------------------------------------------------------------------------
! PURPOSE:
! o given a SCRIP map matrix data file, create a docn/dice domain data file
!
! NOTES:
! o all output data is base on the "_a" grid, the "_b" grid is ignored
! o to compile on an NCAR's SGI, tempest (Dec 2004): 
!   unix> f90 -64 -mips4 -r8 -i4 -lfpe -I/usr/local/include Make_domain.F90 \
!         -L/usr/local/lib64/r4i4 -lnetcdf
!-------------------------------------------------------------------------------

   write(6,*) 'create a dice6/docn6 domain file from a scrip matrix data file'

   !----------------------------------------------------------------------------
   write(6,*) 'input namelist data...'
   !----------------------------------------------------------------------------
   fn_in ='null'
   fn1_out='null'
   fn2_out='null'
   usercomment ='null'
   eps = 1.0e-12
   fminval = 0.001
   fmaxval = c1
   set_fv_pole_yc = 0
   read(*,nml=in_parm)
   write(6,*) 'fn_in  = ',fn_in (1:len_trim(fn_in ))
   write(6,*) 'fn1_out= ',fn1_out(1:len_trim(fn1_out))
   write(6,*) 'fn2_out= ',fn2_out(1:len_trim(fn2_out))
   write(6,*) 'usercomment= ',usercomment(1:len_trim(usercomment))
   write(6,*) 'eps    = ',eps
   write(6,*) 'fminval= ',fminval
   write(6,*) 'fmaxval= ',fmaxval
   write(6,*) 'set_fv_pole_yc = ',set_fv_pole_yc

   !----------------------------------------------------------------------------
   write(6,*) ' '
   write(6,*) 'input SCRIP data...'
   !----------------------------------------------------------------------------

 do nf = 1,2
   if (nf == 1) then
     suffix = '_a'
     fn_out = fn1_out
     complf = .false.
   elseif (nf == 2) then
     suffix = '_b'
     fn_out = fn2_out
     complf = .true.
     pole_fix = .false.
   else
     write(6,*) ' ERROR: nf loop error '
     stop
   endif
   pole_fix = .false.
   if (nf == set_fv_pole_yc) pole_fix = .true.

   write(6,*) ' '
   !--- document global attributes -----------------
   write(6,*) 'input file  = ',fn_in(1:len_trim(fn_in))
   rcode = nf_open(fn_in(1:len_trim(fn_in)),NF_NOWRITE,fid)
   if (rcode.ne.NF_NOERR) write(6,*) nf_strerror(rcode)
   write(6,*) 'open ',trim(fn_in)

!   rcode = nf_get_att_text(fid, NF_GLOBAL, 'title'      , str_title )
!   write(6,*) 'title       = ',str_title (1:len_trim(str_title ))

!   rcode = nf_get_att_text(fid, NF_GLOBAL, 'source'      , str_source)
!   write(6,*) 'source      = ',str_source(1:len_trim(str_source))

   str_da = 'unknown'
   str_db = 'unknown'
   str_grido = 'unknown'
   str_grida = 'unknown'

   rcode = nf_get_att_text(fid, NF_GLOBAL, 'domain_a'     , str_da)
   write(6,*) 'domain_a     = ',str_da(1:len_trim(str_da))
   rcode = nf_get_att_text(fid, NF_GLOBAL, 'domain_b'     , str_db)
   write(6,*) 'domain_b     = ',str_db(1:len_trim(str_db))

   rcode = nf_get_att_text(fid, NF_GLOBAL, 'grid_file_ocn', str_grido)
   write(6,*) 'grid_file_ocn= ',str_grido(1:len_trim(str_grido))
   rcode = nf_get_att_text(fid, NF_GLOBAL, 'grid_file_atm', str_grida)
   write(6,*) 'grid_file_atm= ',str_grida(1:len_trim(str_grida))

   !----------------------------------------------
   ! get domain info for source grid
   !----------------------------------------------
   rcode = nf_inq_dimid (fid, 'n'//trim(suffix) , did)
   rcode = nf_inq_dimlen(fid, did   , n)
   rcode = nf_inq_dimid (fid, 'nv'//trim(suffix), did)
   rcode = nf_inq_dimlen(fid, did   , nv)
   rcode = nf_inq_dimid (fid, 'ni'//trim(suffix), did)
   rcode = nf_inq_dimlen(fid, did   , ni)
   rcode = nf_inq_dimid (fid, 'nj'//trim(suffix), did)
   rcode = nf_inq_dimlen(fid, did   , nj)

   rcode = nf_inq_dimid (fid, 'n_s', did)
   rcode = nf_inq_dimlen(fid, did   , ns)
   rcode = nf_inq_dimid (fid, 'n_a', did)
   rcode = nf_inq_dimlen(fid, did   , na)

   write(6,*) 'n,ni,nj,nv,na,ns=',n,ni,nj,nv,na,ns

   allocate(  xc(   n))   ! x-coordinates of center
   allocate(  yc(   n))   ! y-coordinates of center
   allocate(  xv(nv,n))   ! x-coordinates of verticies
   allocate(  yv(nv,n))   ! y-coordinates of verticies
   allocate(mask(   n))   ! domain mask
   allocate(area(   n))   ! grid cell area
   allocate(frac(   n))   ! area frac of mask "_a" on grid "_b" or float(mask)

   rcode = nf_inq_varid     (fid,'xc'//trim(suffix), vid)
   rcode = nf_get_var_double(fid,vid,  xc )
   rcode = nf_inq_varid     (fid,'yc'//trim(suffix), vid)
   rcode = nf_get_var_double(fid,vid,  yc )
   rcode = nf_inq_varid     (fid,'xv'//trim(suffix), vid)
   rcode = nf_get_var_double(fid,vid,  xv )
   rcode = nf_inq_varid     (fid,'yv'//trim(suffix), vid)
   rcode = nf_get_var_double(fid,vid,  yv )
   rcode = nf_inq_varid     (fid,'mask'//trim(suffix), vid )
   rcode = nf_get_var_int   (fid,vid,mask )
   rcode = nf_inq_varid     (fid,'area'//trim(suffix), vid )
   rcode = nf_get_var_double(fid,vid,area )

   !--- set default frac ---
   frac = c0
   where (mask /= 0) frac = c1

   if (complf) then
      !----------------------------------------------------------------------------
      write(6,*) 'compute frac'
      !----------------------------------------------------------------------------
      allocate(col(ns))
      allocate(row(ns))
      allocate(  S(ns))
      allocate(maska(na))
      allocate(fraca(na))
      rcode = nf_inq_varid     (fid,'col', vid )
      rcode = nf_get_var_int   (fid,vid,col )
      rcode = nf_inq_varid     (fid,'row', vid )
      rcode = nf_get_var_int   (fid,vid,row )
      rcode = nf_inq_varid     (fid,'S', vid )
      rcode = nf_get_var_double(fid,vid,S )
      rcode = nf_inq_varid     (fid,'mask_a', vid )
      rcode = nf_get_var_int   (fid,vid,maska )
      fraca = c0
      where (maska /= 0) fraca = c1
      frac = c0
      !--- compute ocean fraction on atm grid ---
      do k = 1,ns
        frac(row(k)) = frac(row(k)) + fraca(col(k))*S(k)
      enddo
      !--- convert to land fraction, 1.0-frac and ---
      !--- trap errors and modify computed frac ---
      mask(:) = 0
      do k = 1,n
        frac(k) = c1 - frac(k)
	frac_min = min(frac_min,frac(k))
	frac_max = max(frac_max,frac(k))
        if (frac(k) > fmaxval) frac(k) = c1
        if (frac(k) < fminval) frac(k) = c0   ! extra requirement for landfrac
        if (frac(k) /= c0) mask(k) = 1
      enddo
      write(6,*) '----------------------------------------------------------------------'
      write(6,*) 'IMPORTANT: note original min/max frac and decide if that''s acceptable'
      write(6,*) 'original frac clipped above by        : ',fmaxval
      write(6,*) 'original reset to zero when less than : ',fminval
      write(6,*) 'original min, max frac : ',frac_min,frac_max
      write(6,*) 'final min, max frac    : ',minval(frac),maxval(frac)
      write(6,*) '----------------------------------------------------------------------'
   endif

   rcode = nf_close(fid)

   !-----------------------------------------------------------------
   ! adjust j = 1 and j = nj lats to -+ 90 degrees
   !-----------------------------------------------------------------

   if (pole_fix) then
      do i = 1,ni
         yc(i)      = -c90
         yc(n-ni+i) =  c90
      enddo
   endif
 
!
   !----------------------------------------------------------------------------
   write(6,*) ' '
   write(6,*) 'output CCSM data...'
   !----------------------------------------------------------------------------

   !-----------------------------------------------------------------
   ! create a new nc file
   !-----------------------------------------------------------------
   rcode = nf_create(fn_out(1:len_trim(fn_out)),NF_CLOBBER,fid)
   if (rcode.ne.NF_NOERR) write(6,*) nf_strerror(rcode)
   write(6,*) 'write ',trim(fn_out)

   !-----------------------------------------------------------------
   ! global attributes
   !-----------------------------------------------------------------
   str   = 'CCSM domain data: '
   rcode = nf_put_att_text(fid,NF_GLOBAL,'title'      ,len_trim(str),str)

   str   = 'CF-1.0'
   rcode = nf_put_att_text(fid,NF_GLOBAL,'Conventions',len_trim(str),str)

   str = trim(version)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'source_code',len_trim(str),str)

   str = ' $URL: https://svn-ccsm-models.cgd.ucar.edu/tools/mapping/gen_domain/trunk_tags/gen_domain_071001/gen_domain.F90 $'
   rcode = nf_put_att_text(fid,NF_GLOBAL,'SVN_url',len_trim(str),str)

   call date_and_time(cdate,ctime)
   call getenv('LOGNAME',user)
   str = 'created by '//trim(user)//', '//cdate(1:4)//'-'//cdate(5:6)//'-'//cdate(7:8) &
   &                //' '//ctime(1:2)//':'//ctime(3:4)//':'//ctime(5:6)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'history' ,len_trim(str),str)

   str = trim(fn_in)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'source' ,len_trim(str),str)

   str = trim(str_da)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'map_domain_a',len_trim(str),str)

   str = trim(str_db)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'map_domain_b',len_trim(str),str)

   str = trim(str_grido)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'map_grid_file_ocn',len_trim(str),str)

   str = trim(str_grida)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'map_grid_file_atm',len_trim(str),str)

   str = trim(fn1_out)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'output_file1' ,len_trim(str),str)

   str = trim(fn2_out)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'output_file2' ,len_trim(str),str)

   str   = usercomment
   if ( str(1:4) /= 'null' ) &
   & rcode = nf_put_att_text(fid,NF_GLOBAL,'user_comment'   ,len_trim(str),str)

   !-----------------------------------------------------------------
   ! dimension data
   !-----------------------------------------------------------------
   if (n /= ni*nj) STOP 'n'
   rcode = nf_def_dim(fid, 'n' , n , did) ! # of points total
   rcode = nf_def_dim(fid, 'ni', ni, did) ! # of points wrt i
   rcode = nf_def_dim(fid, 'nj', nj, did) ! # of points wrt j
   rcode = nf_def_dim(fid, 'nv',  4, did) ! # of verticies per cell

   !-----------------------------------------------------------------
   ! define data -- coordinates, input grid
   !-----------------------------------------------------------------

   rcode = nf_inq_dimid(fid,'n' , did   )

   rcode = nf_inq_dimid(fid,'ni',vdid(1))
   rcode = nf_inq_dimid(fid,'nj',vdid(2))

   rcode = nf_def_var  (fid,'xc',NF_DOUBLE,2,vdid,vid)
   str   = 'longitude of grid cell center'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'degrees_east'
   rcode = nf_put_att_text(fid,vid,"units"    ,len_trim(str),str)
   str   = 'xv'
   rcode = nf_put_att_text(fid,vid,"bounds"   ,len_trim(str),str)

   rcode = nf_def_var  (fid,'yc',NF_DOUBLE,2,vdid,vid)
   str   = 'latitude of grid cell center'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'degrees_north'
   rcode = nf_put_att_text(fid,vid,"units"    ,len_trim(str),str)
   str   = 'yv'
   rcode = nf_put_att_text(fid,vid,"bounds"   ,len_trim(str),str)
   if (pole_fix) then
      write(str,*) 'set_fv_pole_yc ON, yc = -+90 at j=1,j=nj'
      rcode = nf_put_att_text(fid,vid,'filter1' ,len_trim(str),str)
   endif

   rcode = nf_inq_dimid(fid,'nv',vdid(1))
   rcode = nf_inq_dimid(fid,'ni',vdid(2))
   rcode = nf_inq_dimid(fid,'nj',vdid(3))

   rcode = nf_def_var  (fid,'xv',NF_DOUBLE,3,vdid,vid)
   str   = 'longitude of grid cell verticies'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'degrees_east'
   rcode = nf_put_att_text(fid,vid,"units"    ,len_trim(str),str)

   rcode = nf_def_var  (fid,'yv',NF_DOUBLE,3,vdid,vid)
   str   = 'latitude of grid cell verticies'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'degrees_north'
   rcode = nf_put_att_text(fid,vid,"units"    ,len_trim(str),str)

   rcode = nf_inq_dimid(fid,'ni',vdid(1))
   rcode = nf_inq_dimid(fid,'nj',vdid(2))

   rcode = nf_def_var  (fid,'mask',NF_INT ,2,vdid,vid)
   str   = 'domain mask'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'unitless'
   rcode = nf_put_att_text(fid,vid,"note"    ,len_trim(str),str)
   str   = 'xc yc'
   rcode = nf_put_att_text(fid,vid,"coordinates",len_trim(str),str)
   str   = '0 value indicates cell is not active'
   rcode = nf_put_att_text(fid,vid,"comment",len_trim(str),str)

   rcode = nf_def_var  (fid,'area',NF_DOUBLE,2,vdid,vid)
   str   = 'area of grid cell in radians squared'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'xc yc'
   rcode = nf_put_att_text(fid,vid,"coordinates",len_trim(str),str)
   str   = 'radian2'
   rcode = nf_put_att_text(fid,vid,"units"    ,len_trim(str),str)

   rcode = nf_def_var  (fid,'frac',NF_DOUBLE ,2,vdid,vid)
   str   = 'fraction of grid cell that is active'
   rcode = nf_put_att_text(fid,vid,"long_name",len_trim(str),str)
   str   = 'xc yc'
   rcode = nf_put_att_text(fid,vid,"coordinates",len_trim(str),str)
   str   = 'unitless'
   rcode = nf_put_att_text(fid,vid,"note"     ,len_trim(str),str)
   write(str,'(a,g14.7)') 'error if frac> 1.0+eps or frac < 0.0-eps; eps =',eps
   rcode = nf_put_att_text(fid,vid,'filter1' ,len_trim(str),str)
   write(str,'(a,g14.7,a,g14.7)') 'limit frac to [fminval,fmaxval]; fminval=',fminval,' fmaxval=',fmaxval
   rcode = nf_put_att_text(fid,vid,'filter2' ,len_trim(str),str)

   rcode = nf_enddef(fid)

   rcode = nf_inq_varid     (fid,  'xc',vid)
   rcode = nf_put_var_double(fid,  vid , xc)
   rcode = nf_inq_varid     (fid,  'yc',vid)
   rcode = nf_put_var_double(fid,  vid , yc)
   rcode = nf_inq_varid     (fid,  'xv',vid)
   rcode = nf_put_var_double(fid,  vid , xv)
   rcode = nf_inq_varid     (fid,  'yv',vid)
   rcode = nf_put_var_double(fid,  vid , yv)
   rcode = nf_inq_varid     (fid,'mask',vid)
   rcode = nf_put_var_int   (fid,  vid ,mask)
   rcode = nf_inq_varid     (fid,'area',vid)
   rcode = nf_put_var_double(fid,  vid ,area)
   rcode = nf_inq_varid     (fid,'frac',vid)
   rcode = nf_put_var_double(fid,  vid ,frac)

   rcode = nf_close(fid)
   write(6,*) 'close output file'

   if (rcode.ne.NF_NOERR) write(6,*) nf_strerror(rcode)
enddo

END PROGRAM

!===============================================================================
