program paleo_mkraw

  implicit none
  include 'netcdf.inc'

!----------------------------------------------------------------- 
!
! make raw datasets for Paleo CCSM simulation
! input data: (2x2 ascii) LSM vegetation types from desired period
!           : (mksrf_soitex.10level.nc) IGBP soil texture for 0k
!           : input file pathname specified below
! output data: raw data files necessary to create surface-data file
!              at model runtime: mksrf_glacier.nc
!			       : mksrf_urban.nc
!			       : mksrf_lanwat.nc
!			       : mksrf_soicol_clm2.nc 
!			       : mksrf_pft.nc
!			       : mksrf_lai.nc
!			       : mksrf_soitex.10level_paleo.nc
!            : output file names specifed below
!
! from 2x2 LSM veg type data, this program will create pfts, glacier,
! urban, lanwat, soilcol, lai  fields. the soil texture fields will
! place soil profiles on
! the paleo grid. color will be assigned to a value of 4.
! glacier,urban,and lakes will all be set to zero. 
!
! this program is designed to provide a framework to create the
! raw data files. any modifications to the aligorthims in this code
! can be applied by the user appropriate for the time period. 
! 
! cshields, slevis,  feb 2003
!-----------------------------------------------------------------


  integer, parameter :: r8 = selected_real_kind(12)

  integer, parameter :: nlon = 180        !input grid : longitude points
  integer, parameter :: nlat =  90        !input grid : latitude  points
  integer, parameter :: numpft = 16       !number of plant types
  integer, parameter :: time   = 12       !number of months in year 
  integer, parameter :: nlay = 10         !input grid : number of soil layers
 
  integer, parameter :: nlon_st = 4320      !soil input grid : longitude points
  integer, parameter :: nlat_st = 2160      !soil input grid : latitude  points
  integer, parameter :: nmapunits = 4931    !soil input grid : # of igbp soil 'mapunits'
  integer, parameter :: mapunitmax = 6998   !soil input grid : max value of 'mapunits'


  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid
  real(r8) :: dx,dy                       !grid increments
  real*4   :: ilon(nlon)                  !longitude dimension array (1d)
  real*4   :: ilat(nlat)                  !latitude dimension array (1d) 

  real(r8) :: landmask(nlon,nlat)         !fraction of land
  real(r8) :: dzsoi(10), zsoi(10)         !soil layer thickness and depth
  real(r8) :: pct_sand0(nlay,mapunitmax)  !original percent sand 
  real(r8) :: pct_clay0(nlay,mapunitmax)  !original percent clay 
  real(r8) :: mapunits0(nlon_st,nlat_st)  !mapunits 

  real(r8) :: pct_pft(nlon,nlat,0:numpft) !percent pft 
  real(r8) :: pct_glacier(nlon,nlat)      !percent glacier
  real(r8) :: pct_urban(nlon,nlat)        !percent urban 
  real(r8) :: pct_wetland(nlon,nlat)      !percent wetland 
  real(r8) :: pct_lake(nlon,nlat)         !percent lake 
  real(r8) :: sumpctland(nlon,nlat)       !percent lake 
 

  integer :: i,j,m,ret                    !indices
  integer :: ndata = 1                    !input unit
  integer :: ncid_lsm_i                   !netCDF file id for lsm input
  integer :: ncid_soitex_i                !netCDF file id for soitex input
  integer :: ncid_glacier                 !netCDF file id for glacier output
  integer :: ncid_urban                   !netCDF file id for urban " 
  integer :: ncid_lanwat                  !netCDF file id for lanwat "
  integer :: ncid_lai                     !netCDF file id for lai "
  integer :: ncid_pft                     !netCDF file id for pft  "
  integer :: ncid_soicol                  !netCDF file id for soicol " 
  integer :: ncid_soitex                  !netCDF file id for soitex "

  integer :: veg_id		          !veg id
  integer :: lon_id		          !lon id
  integer :: lat_id		          !lat id
  integer :: dzsoi_id		          !dzsoi id
  integer :: zsoi_id		          !zsoi id
  integer :: pct_clay0_id	          !pct_clay0 id
  integer :: pct_sand0_id                 !pct_sand0 id
  integer :: mapunits0_id                  !mapunits id

  integer :: iveg(nlon,nlat)	          !lsm veg array  (input)
  integer ::  veg(nlon,nlat)	          !lsm veg array  (re-ordered input, if necessary)
  integer :: londum(nlon)                 !lon integer on ascii file
  integer :: latdum(nlat)                 !lat integer on ascii file



  character(len=80) :: file_format		  ! input filenames
  character(len=80) :: filei_lsmveg,filei_soitex  ! input filenames
  character(len=80) :: fileo_glacier              !output filenames
  character(len=80) :: fileo_urban                !output filenames
  character(len=80) :: fileo_lanwat               !output filenames
  character(len=80) :: fileo_lai                  !output filenames
  character(len=80) :: fileo_pft                  !output filenames
  character(len=80) :: fileo_soicol               !output filenames
  character(len=80) :: fileo_soitex               !output filenames
 
  character(len=78) :: char_string        !character string for ascii file


!-----------------------------------------------------------------

! Determine input and output file names

  file_format  =  'infile_format'
  filei_lsmveg =  'input_lsm_data'
  filei_soitex =  'input_soi_data'

  fileo_glacier= 'output_glacier'
  fileo_urban  = 'output_urban' 
  fileo_lanwat = 'output_lanwat'
  fileo_lai    = 'output_lai'
  fileo_pft    = 'output_pft'
  fileo_soicol = 'output_soicol'
  fileo_soitex = 'output_soitex'

  print *,'Output file names defined '

! -----------------------------------------------------------------
! Determine grid for ascii input data 
!
! All data needs to be written as follows:
!       a) it should have a 78 character descriptor
!       b) long should be written from -179 to +179 (centered)
!       c) lat  should be written from -89  to + 89 (centered)
!
! 2.0 x 2.0 degree, stored in latitude bands,
! from south to north. In a given latitude band, data begin
! at the dateline and proceed eastward.
! -----------------------------------------------------------------

! Define North, East, South, West edges of grid

  edge(1) =   90.
  edge(2) =  180.
  edge(3) =  -90.
  edge(4) = -180.

! Make latitudes and longitudes at center of grid cell

  dx = (edge(2)-edge(4)) / nlon
  dy = (edge(1)-edge(3)) / nlat

  do j = 1, nlat
     do i = 1, nlon
        latixy(i,j) = (edge(3)+dy/2.) + (j-1)*dy
        longxy(i,j) = (edge(4)+dx/2.) + (i-1)*dx
       end do
  end do

  lat(:) = latixy(1,:)
  lon(:) = longxy(:,1)

  print *, 'LAT= ',lat
  print *, 'LON= ',lon

  print *, 'Lat and Lon arrays defined '

! -------------------------------------------------------------------
! Read in netcdf LSM vegetation data
! -------------------------------------------------------------------
! -----------------------------------------------------------------
! Determine grid for netcdf input data 
!
! netcdf data needs to be written as follows:
!       a) netcdf format should match the topo/bathy format
!       b) lon should be written from   1 to 360 (centered)
!       c) lat should be written from -89 to  89 (centered)
! -----------------------------------------------------------------

if (trim(file_format) == 'netcdf') then
!
  ret = nf_open (filei_lsmveg, nf_nowrite, ncid_lsm_i)
  if (ret == nf_noerr) then
    write(6,*)'Successfully opened netcdf LSM veg types ',trim(filei_lsmveg)
    call wrap_inq_varid (ncid_lsm_i, 'veg', veg_id   )
    call wrap_inq_varid (ncid_lsm_i, 'lon', lon_id   )
    call wrap_inq_varid (ncid_lsm_i, 'lat', lat_id   )
    call wrap_get_flt  (ncid_lsm_i, lon_id, ilon) 
    call wrap_get_flt  (ncid_lsm_i, lat_id, ilat) 
    call wrap_get_int  (ncid_lsm_i, veg_id, iveg) 
  else
    write(6,*)'cannot open lsm veg file successfully'
    call endrun
  endif
  ret = nf_close (ncid_lsm_i)
  print *, '2x2 Netcdf LSM Vegetation Types file read '

! -------------------------------------------------------------------
! Re-order iveg from 0-360 => -179-179 and create land mask
! -------------------------------------------------------------------
  do j = 1, nlat
  do i = 1, nlon
	if (i < 91) then
		veg(i,j) = iveg(i+90,j)
		! print *, 'input lon ',ilon(i+90),'Reference lon  ', ilon(i+90)-360, 'mapped lon ', lon(i),'i ',i
	else	     
		veg(i,j) = iveg(i-90,j)
		! print *, 'input lon ', ilon(i-90),' Reference lon ', ilon(i-90)    , 'mapped lon ', lon(i),'i ',i
		
	endif
   	if (veg(i,j)/=0) landmask(i,j) = 1.
  end do
  end do

! -------------------------------------------------------------------
! Read in ASCII formatted LSM vegetation data
! -------------------------------------------------------------------
else
! -----------------------------------------------------------------
! Determine grid for ascii input data 
!
! All ascii data needs to be written as follows:
!       a) it should have a 78 character descriptor
!       b) long should be written from -179 to +179 (centered)
!       c) lat  should be written from -89  to + 89 (centered)
!
! 2.0 x 2.0 degree, stored in latitude bands,
! from south to north. In a given latitude band, data begin
! at the dateline and proceed eastward.
! -----------------------------------------------------------------

  open(ndata,file=filei_lsmveg,status='old',form='formatted')

  read(ndata,100) char_string
  print *,  'ASCII file= ',char_string
  read(ndata,101) (londum(i),i=1,nlon)
  do j = 1, nlat
   read(ndata,102) latdum(j),(veg(i,j),i=1,nlon)
  enddo

  do j = 1, nlat
  do i = 1, nlon
   if (veg(i,j)/=0) landmask(i,j) = 1.
  end do
  end do

100     format(a78)
101     format(180(i4,1x))
102     format(i3,5x,180(i2))

  close(ndata)

  print *, '2x2 Ascii LSM Vegetation types read '

endif

! -------------------------------------------------------------------
! Read in netcdf soil texture file
! -------------------------------------------------------------------
!
  ret = nf_open (filei_soitex, nf_nowrite, ncid_soitex_i)
  if (ret == nf_noerr) then
    write(6,*)'Successfully opened netcdf soitex ',trim(filei_soitex)
    call wrap_inq_varid (ncid_soitex_i, 'DZSOI', dzsoi_id   )
    call wrap_inq_varid (ncid_soitex_i, 'ZSOI', zsoi_id   )
    call wrap_inq_varid (ncid_soitex_i, 'PCT_CLAY', pct_clay0_id   )
    call wrap_inq_varid (ncid_soitex_i, 'PCT_SAND', pct_sand0_id   )
    call wrap_inq_varid (ncid_soitex_i, 'MAPUNITS', mapunits0_id   )
    call wrap_get_var8 (ncid_soitex_i, dzsoi_id, dzsoi) 
    call wrap_get_var8 (ncid_soitex_i, zsoi_id, zsoi)
    call wrap_get_var8 (ncid_soitex_i, pct_clay0_id, pct_clay0)
    call wrap_get_var8 (ncid_soitex_i, pct_sand0_id, pct_sand0)
    call wrap_get_var8 (ncid_soitex_i, mapunits0_id, mapunits0)
  else
    write(6,*)'cannot open soil texture file successfully'
    call endrun
  endif
  ret = nf_close (ncid_soitex_i)
  print *, 'Netcdf IGBC Soil Texture file read '


! -------------------------------------------------------------------
! Call subroutines to create each raw data file
! --------------------------------------------------------------------

  call create_mksrf_glacier(veg,landmask,                         &
      		            nlon,nlat,lon,longxy,lat,latixy,edge, &
		            fileo_glacier,ncid_glacier,	          &
		            pct_glacier)
  print *, 'Netcdf File mksfr_glacier_paleo created'

  call create_mksrf_urban(veg,landmask,			          &
	 		  nlon,nlat,lon,longxy,lat,latixy,edge,   &
	                  fileo_urban,ncid_urban,		  &
		          pct_urban)
  print *, 'Netcdf File mksfr_urban_paleo created'

  call create_mksrf_lanwat(veg,landmask,		          &
	             	   nlon,nlat,lon,longxy,lat,latixy,edge,  &
  		           fileo_lanwat,ncid_lanwat,		  &
			   pct_wetland,pct_lake)
  print *, 'Netcdf File mksfr_lanwat_paleo created'

  call create_mksrf_pft(veg,landmask,numpft,			  &
	                nlon,nlat,lon,longxy,lat,latixy,edge,     &
		        fileo_pft,ncid_pft,                       &
                        pct_pft) 
  print *, 'Netcdf File mksfr_pft_paleo created'

  call create_mksrf_lai(veg,landmask,numpft,		 	  &
                        nlon,nlat,lon,longxy,lat,latixy,edge,     &
 			fileo_lai,ncid_lai,                       &
                        pct_pft) 
  print *, 'Netcdf File mksfr_lai_paleo created'

  call create_mksrf_soicol(veg,landmask,			  &
			   nlon,nlat,lon,longxy,lat,latixy,edge , &
		           fileo_soicol,ncid_soicol)
  print *, 'Netcdf File mksfr_soicol_paleo created'

  call create_mksrf_soitex(veg,landmask,            	             &
		           dzsoi,zsoi,mapunits0,pct_clay0,pct_sand0, &
			   nlay,nlon_st,nlat_st,nmapunits,mapunitmax,&
		           nlon,nlat,lon,longxy,lat,latixy,edge,     &
                           fileo_soitex,ncid_soitex)
  print *, 'Netcdf File mksfr_soitex_paleo created'

! error check 
! pct_glacier+pct_lake+pct_wetland+pct_urban and checks .le. 100.
  sumpctland = pct_glacier+pct_lake+pct_wetland+pct_urban 
  do j = 1, nlat
    do i = 1, nlon
      if (landmask(i,j)==1 .and. sumpctland(i,j)>100.) then
        write(*,*) 'ERROR: sumpctland (glacier+urban+wetland+lake)= ',&
	            sumpctland(i,j),i,j
      end if
    end do
  end do

  print *, 'End program' 

end program paleo_mkraw




!===============================================================================
subroutine create_mksrf_glacier(veg,landmask,			      &
		                nlon,nlat,lon,longxy,lat,latixy,edge, &
			        fileo,ncid)

  implicit none
  include 'netcdf.inc'


! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid

  integer :: nlon,nlat,ncid              !number lats/lons, nc fileo id 
  character(len=80) :: fileo             !output filenames

! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------

  real(r8) :: pct_glacier(nlon,nlat)      !percent glacier

  integer :: dimlon_id                    !netCDF dimension id
  integer :: dimlat_id                    !netCDF dimension id

  integer :: lon_id                       !1d longitude array id
  integer :: lat_id                       !1d latitude array id
  integer :: longxy_id                    !2d longitude array id
  integer :: latixy_id                    !2d latitude array id
  integer :: edgen_id                     !northern edge of grid (edge(1)) id
  integer :: edgee_id                     !eastern  edge of grid (edge(2)) id
  integer :: edges_id                     !southern edge of grid (edge(3)) id
  integer :: edgew_id                     !western  edge of grid (edge(4)) id
  integer :: pct_glacier_id               !pct_glacier id
  integer :: landmask_id                  !landmask id

  integer :: i,j                          !indices
  integer :: dim1_id(1)                   !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)                   !netCDF dimension id for 2-d variables
  integer :: status                       !status

  character(len=256) :: name,unit            !netCDF attributes


! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes

  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'pct_glacier_data')

! Define dimensions

  call wrap_def_dim (ncid, 'lon' , nlon, dimlon_id)
  call wrap_def_dim (ncid, 'lat' , nlat, dimlat_id)

! Define grid variables

  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)

  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)

  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)

  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)

  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)

  name = 'western edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)

! Define input file specific variables

  name = 'percent glacier'
  unit = 'unitless'
  dim2_id(1) = lon_id
  dim2_id(2) = lat_id
  call wrap_def_var (ncid ,'PCT_GLACIER' ,nf_float, 2, dim2_id, pct_glacier_id)
  call wrap_put_att_text (ncid, pct_glacier_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_glacier_id, 'units'    , unit)

  name = 'land mask'
  unit = 'unitless'
  call wrap_def_var (ncid ,'LANDMASK' ,nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

! End of definition

  status = nf_enddef(ncid)

! -----------------------------------------------------------------------
! Create pct_glacier from LSM vegtypes 
! -----------------------------------------------------------------------

  do j = 1, nlat
    do i = 1, nlon
      if (veg(i,j).eq.1) then
        pct_glacier(i,j) = 100._r8
      else
        pct_glacier(i,j) = 0._r8
      end if
    end do
  end do

! --------------------------------------------------------------------------
! Write variables
! --------------------------------------------------------------------------

  call wrap_put_var_realx (ncid, lon_id        , lon)
  call wrap_put_var_realx (ncid, lat_id        , lat)
  call wrap_put_var_realx (ncid, longxy_id     , longxy)
  call wrap_put_var_realx (ncid, latixy_id     , latixy)
  call wrap_put_var_realx (ncid, edgen_id      , edge(1))
  call wrap_put_var_realx (ncid, edgee_id      , edge(2))
  call wrap_put_var_realx (ncid, edges_id      , edge(3))
  call wrap_put_var_realx (ncid, edgew_id      , edge(4))
  call wrap_put_var_realx (ncid, pct_glacier_id, pct_glacier)
  call wrap_put_var_realx (ncid, landmask_id   , landmask)

  call wrap_close(ncid)

end subroutine create_mksrf_glacier



!========================================================================
subroutine create_mksrf_urban(veg,landmask,      	            &
			      nlon,nlat,lon,longxy,lat,latixy,edge, &
			      fileo,ncid) 

  implicit none
  include 'netcdf.inc'

! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid

  integer :: nlon,nlat,ncid              !number lats/lons, nc fileo id 
  character(len=80) :: fileo             !output filenames

! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------

  real(r8) :: pct_urban(nlon,nlat)        !pct urban

  integer :: dimlon_id                    !netCDF dimension id
  integer :: dimlat_id                    !netCDF dimension id

  integer :: lon_id                       !1d longitude array id
  integer :: lat_id                       !1d latitude array id
  integer :: longxy_id                    !2d longitude array id
  integer :: latixy_id                    !2d latitude array id
  integer :: edgen_id                     !northern edge of grid (edge(1)) id
  integer :: edgee_id                     !eastern  edge of grid (edge(2)) id
  integer :: edges_id                     !southern edge of grid (edge(3)) id
  integer :: edgew_id                     !western  edge of grid (edge(4)) id
  integer :: pct_urban_id                 !percent urban id
  integer :: landmask_id                  !landmask id

  integer :: i,j                          !indices
  integer :: dim1_id(1)                   !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)                   !netCDF dimension id for 2-d variables
  integer :: status                       !status

  character(len=256) :: name,unit         !netCDF attributes

! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes

  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'pct_urban_data')

! Define dimensions

  call wrap_def_dim (ncid, 'lon' , nlon, dimlon_id)
  call wrap_def_dim (ncid, 'lat' , nlat, dimlat_id)

! Define input file independent variables 

  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)

  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)

  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)

  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)

  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)

  name = 'western edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)

! Define input file specific variables

  name = 'percent urban'
  unit = 'unitless'
  dim2_id(1) = lon_id
  dim2_id(2) = lat_id
  call wrap_def_var (ncid ,'PCT_URBAN' ,nf_float, 2, dim2_id, pct_urban_id)
  call wrap_put_att_text (ncid, pct_urban_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_urban_id, 'units'    , unit)

  name = 'land mask'
  unit = 'unitless'
  call wrap_def_var (ncid ,'LANDMASK' ,nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

! End of definition

  status = nf_enddef(ncid)


! -----------------------------------------------------------------------
! Create pct_urban from LSM vegtypes
! -----------------------------------------------------------------------

  pct_urban = 0.

! --------------------------------------------------------------------------
! Write variables
! ----------------------------------------------------------------------

  call wrap_put_var_realx (ncid, lon_id      , lon)
  call wrap_put_var_realx (ncid, lat_id      , lat)
  call wrap_put_var_realx (ncid, longxy_id   , longxy)
  call wrap_put_var_realx (ncid, latixy_id   , latixy)
  call wrap_put_var_realx (ncid, edgen_id    , edge(1))
  call wrap_put_var_realx (ncid, edgee_id    , edge(2))
  call wrap_put_var_realx (ncid, edges_id    , edge(3))
  call wrap_put_var_realx (ncid, edgew_id    , edge(4))
  call wrap_put_var_realx (ncid, pct_urban_id, pct_urban)
  call wrap_put_var_realx (ncid, landmask_id, landmask)

  call wrap_close(ncid)

end subroutine create_mksrf_urban



!===========================================================================
subroutine create_mksrf_lanwat(veg,landmask,			     &
			       nlon,nlat,lon,longxy,lat,latixy,edge, &
                               fileo,ncid)
 

  implicit none
  include 'netcdf.inc'

! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid

  integer :: nlon,nlat,ncid              !number lats/lons, nc fileo id 
  character(len=80) :: fileo             !output filenames

! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------


  real(r8) :: pct_lake(nlon,nlat)         !pct lake
  real(r8) :: pct_wetland(nlon,nlat)      !pct wetland

  integer :: dimlon_id                    !netCDF dimension id
  integer :: dimlat_id                    !netCDF dimension id

  integer :: lon_id                       !1d longitude array id
  integer :: lat_id                       !1d latitude array id
  integer :: longxy_id                    !2d longitude array id
  integer :: latixy_id                    !2d latitude array id
  integer :: edgen_id                     !northern edge of grid (edge(1)) id
  integer :: edgee_id                     !eastern  edge of grid (edge(2)) id
  integer :: edges_id                     !southern edge of grid (edge(3)) id
  integer :: edgew_id                     !western  edge of grid (edge(4)) id
  integer :: pct_lake_id                  !pct_lake id
  integer :: pct_wetland_id               !pct_wetland id
  integer :: landmask_id                  !landmask id

  integer :: i,j                          !indices
  integer :: dim1_id(1)                   !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)                   !netCDF dimension id for 2-d variables
  integer :: status                       !status

  character(len=256) :: name,unit         !netCDF attributes


! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes

  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'lanwat_data')

! Define dimensions

  call wrap_def_dim (ncid, 'lon' , nlon, dimlon_id)
  call wrap_def_dim (ncid, 'lat' , nlat, dimlat_id)

! Define input file independent variables 

  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)

  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)

  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)

  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)

  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)

  name = 'western edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)

! Define percent lake and wetland variables

  name = 'percent lake'
  unit = 'unitless'
  dim2_id(1) = lon_id
  dim2_id(2) = lat_id
  call wrap_def_var (ncid ,'PCT_LAKE' ,nf_float, 2, dim2_id, pct_lake_id)
  call wrap_put_att_text (ncid, pct_lake_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_lake_id, 'units'    , unit)

  name = 'percent wetland'
  unit = 'unitless'
  dim2_id(1) = lon_id
  dim2_id(2) = lat_id
  call wrap_def_var (ncid ,'PCT_WETLAND' ,nf_float, 2, dim2_id, pct_wetland_id)
  call wrap_put_att_text (ncid, pct_wetland_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_wetland_id, 'units'    , unit)

  name = 'land mask'
  unit = 'unitless'
  call wrap_def_var (ncid ,'LANDMASK' ,nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

! End of definition

  status = nf_enddef(ncid)


! -----------------------------------------------------------------------
! Create pct_lake and pct_wetland from LSM vegtypes 
! -----------------------------------------------------------------------

 pct_lake = 0.

 do j = 1, nlat
   do i = 1, nlon
     if (veg(i,j).eq.27) then
       pct_wetland(i,j) = 20._r8
     elseif (veg(i,j).eq.28) then
       pct_wetland(i,j) = 100._r8
     else
       pct_wetland(i,j) = 0._r8
     end if
   end do
 end do

! --------------------------------------------------------------------------
! Write variables
! --------------------------------------------------------------------------


  call wrap_put_var_realx (ncid, lon_id        , lon)
  call wrap_put_var_realx (ncid, lat_id        , lat)
  call wrap_put_var_realx (ncid, longxy_id     , longxy)
  call wrap_put_var_realx (ncid, latixy_id     , latixy)
  call wrap_put_var_realx (ncid, edgen_id      , edge(1))
  call wrap_put_var_realx (ncid, edgee_id      , edge(2))
  call wrap_put_var_realx (ncid, edges_id      , edge(3))
  call wrap_put_var_realx (ncid, edgew_id      , edge(4))
  call wrap_put_var_realx (ncid, pct_lake_id   , pct_lake)
  call wrap_put_var_realx (ncid, pct_wetland_id, pct_wetland)
  call wrap_put_var_realx (ncid, landmask_id   , landmask)

  call wrap_close(ncid)

end subroutine create_mksrf_lanwat



!=============================================================================
subroutine create_mksrf_pft(veg,landmask,numpft,		  &
			    nlon,nlat,lon,longxy,lat,latixy,edge, &
			    fileo,ncid,                           &
                            pct_pft) 

  implicit none
  include 'netcdf.inc'

! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid

  integer :: nlon,nlat,numpft,ncid       !number lats/lons/pfts, nc fileo id 
  character(len=80) :: fileo             !output filenames

  real(r8) :: pct_pft(nlon,nlat,0:numpft) !percent pft 

! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------

  real(r8) :: sumpctpft(nlon,nlat)        
  integer :: dimlon_id                    !netCDF dimension id
  integer :: dimlat_id                    !netCDF dimension id
  integer :: dimpft_id                    !netCDF dimension id

  integer :: lon_id                       !1d longitude array id
  integer :: lat_id                       !1d latitude array id
  integer :: longxy_id                    !2d longitude array id
  integer :: latixy_id                    !2d latitude array id
  integer :: edgen_id                     !northern edge of grid (edge(1)) id
  integer :: edgee_id                     !eastern  edge of grid (edge(2)) id
  integer :: edges_id                     !southern edge of grid (edge(3)) id
  integer :: edgew_id                     !western  edge of grid (edge(4)) id
  integer :: landmask_id                  !landmask id
  integer :: pct_pft_id                   !pct_pft id

  integer :: i,j                          !indices
  integer :: dim1_id(1)                   !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)                   !netCDF dimension id for 2-d variables
  integer :: dim3_id(3)                   !netCDF dimension id for 3-d variables
  integer :: status                       !status

  character(len=256) :: name,unit         !netCDF attributes
  
! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes

  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'pft_data')

! Define dimensions

  call wrap_def_dim (ncid, 'lon', nlon    , dimlon_id)
  call wrap_def_dim (ncid, 'lat', nlat    , dimlat_id)
  call wrap_def_dim (ncid, 'pft', numpft+1, dimpft_id)

! Define grid variables

  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)

  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)

  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)

  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)

  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)

  
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)

! Define pft variables

  name = 'land mask'
  unit = 'unitless'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid ,'LANDMASK' ,nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

  name = 'percent pft'
  unit = 'unitless'
  dim3_id(1) = dimlon_id
  dim3_id(2) = dimlat_id
  dim3_id(3) = dimpft_id
  call wrap_def_var (ncid ,'PCT_PFT' ,nf_float, 3, dim3_id, pct_pft_id)
  call wrap_put_att_text (ncid, pct_pft_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_pft_id, 'units'    , unit)

! End of definitions

  status = nf_enddef(ncid)

! -----------------------------------------------------------------------
! Create pfts from LSM vegtypes 
! -----------------------------------------------------------------------

  pct_pft = 0.                        !initialize
  do j = 1, nlat
    do i = 1, nlon
      if (veg(i,j).eq.1) then         !land ice
        pct_pft(i,j,0) = 100._r8      !is bare
        pct_pft(i,j,1:numpft) = 0._r8
      elseif (veg(i,j).eq.2) then     !desert
        pct_pft(i,j,0) = 100._r8      !is bare
      elseif (veg(i,j).eq.3) then     !cool needleleaf evergreen forest
        pct_pft(i,j,0) = 25._r8       !includes bare ground and
        pct_pft(i,j,2) = 75._r8       !needleleaf evergreen boreal tree
      elseif (veg(i,j).eq.4) then     !cool needleleaf deciduous forest
        pct_pft(i,j,0) = 50._r8       !includes bare ground and
        pct_pft(i,j,3) = 50._r8       !needleleaf deciduous boreal tree
      elseif (veg(i,j).eq.5) then     !cool broadleaf deciduous forest
        pct_pft(i,j,0) = 25._r8       !includes bare ground and
        pct_pft(i,j,8) = 75._r8       !broadleaf deciduous boreal tree
      elseif (veg(i,j).eq.6) then     !cool mixed forest
        pct_pft(i,j,0) = 26._r8       !includes bare ground and
        pct_pft(i,j,2) = 37._r8       !needleleaf evergreen boreal tree
        pct_pft(i,j,8) = 37._r8       !broadleaf deciduous boreal tree
      elseif (veg(i,j).eq.7) then     !warm needleleaf evergreen forest
        pct_pft(i,j,0) = 25._r8       !includes bare ground and
        pct_pft(i,j,1) = 75._r8       !needleleaf evergreen temperate tree
      elseif (veg(i,j).eq.8) then     !warm broadleaf deciduous forest
        pct_pft(i,j,0) = 25._r8       !includes bare ground and
        pct_pft(i,j,7) = 75._r8       !broadleaf deciduous temperate tree
      elseif (veg(i,j).eq.9) then     !warm mixed forest
        pct_pft(i,j,0) = 26._r8       !includes bare ground and
        pct_pft(i,j,1) = 37._r8       !needleleaf evergreen temperate tree
        pct_pft(i,j,7) = 37._r8       !broadleaf deciduous temperate tree
      elseif (veg(i,j).eq.10) then    !tropical broadleaf evergreen forest
        pct_pft(i,j,0) =  5._r8       !includes bare ground and
        pct_pft(i,j,4) = 95._r8       !broadleaf evergreen tropical tree
      elseif (veg(i,j).eq.11) then    !tropical seasonal deciduous forest
        pct_pft(i,j,0) = 25._r8       !includes bare ground and
        pct_pft(i,j,6) = 75._r8       !broadleaf deciduous tropical tree
      elseif (veg(i,j).eq.12) then    !savanna
        pct_pft(i,j,14) = 70._r8      !includes warm c4 grass and
        pct_pft(i,j,6) = 30._r8       !broadleaf deciduous tropical tree
      elseif (veg(i,j).eq.13) then    !evergreen forest tundra
        pct_pft(i,j,0) = 50._r8       !includes bare ground and
        pct_pft(i,j,2) = 25._r8       !needleleaf evergreen boreal tree
        pct_pft(i,j,12) = 25._r8      !arctic c3 grass
      elseif (veg(i,j).eq.14) then    !deciduous forest tundra
        pct_pft(i,j,0) = 50._r8       !includes bare ground and
        pct_pft(i,j,3) = 25._r8       !needleleaf deciduous boreal tree
        pct_pft(i,j,12) = 25._r8      !arctic c3 grass
      elseif (veg(i,j).eq.15) then    !cool forest crop
        pct_pft(i,j,15) = 40._r8      !crop
        pct_pft(i,j,2) = 30._r8       !needleleaf evergreen boreal tree
        pct_pft(i,j,8) = 30._r8       !broadleaf deciduous boreal tree
      elseif (veg(i,j).eq.16) then    !warm forest crop
        pct_pft(i,j,15) = 40._r8      !crop
        pct_pft(i,j,1) = 30._r8       !needleleaf evergreen temperate tree
        pct_pft(i,j,7) = 30._r8       !broadleaf deciduous temperate tree
      elseif (veg(i,j).eq.17) then    !cool grassland
        pct_pft(i,j,0) = 20._r8       !includes bare ground and
        pct_pft(i,j,13) = 60._r8      !cool c3 grass
        pct_pft(i,j,14) = 20._r8      !warm c4 grass
      elseif (veg(i,j).eq.18) then    !warm grassland
        pct_pft(i,j,0) = 20._r8       !includes bare ground and
        pct_pft(i,j,13) = 20._r8      !cool c3 grass
        pct_pft(i,j,14) = 60._r8      !warm c4 grass
      elseif (veg(i,j).eq.19) then    !tundra
        pct_pft(i,j,0) = 40._r8       !includes bare ground and
        pct_pft(i,j,11) = 30._r8      !broadleaf deciduous boreal shrub
        pct_pft(i,j,12) = 30._r8      !arctic c3 grass
      elseif (veg(i,j).eq.20) then    !evergreen shrubland
        pct_pft(i,j,0) = 20._r8       !includes bare ground and
        pct_pft(i,j,9) = 80._r8       !broadleaf evergreen temperate shrub
      elseif (veg(i,j).eq.21) then    !deciduous shrubland
        pct_pft(i,j,0) = 20._r8       !includes bare ground and
        pct_pft(i,j,10) = 80._r8      !broadleaf deciduous temperate shrub
      elseif (veg(i,j).eq.22) then    !semi-desert
        pct_pft(i,j,0) = 90._r8       !includes bare ground and
        pct_pft(i,j,10) = 10._r8      !broadleaf deciduous temperate shrub
      elseif (veg(i,j).eq.23) then    !cool irrigated crop
        pct_pft(i,j,0) = 15._r8       !includes bare ground and
        pct_pft(i,j,15) = 85._r8      !crop
      elseif (veg(i,j).eq.24) then    !cool crop
        pct_pft(i,j,0) = 15._r8       !includes bare ground and
        pct_pft(i,j,15) = 85._r8      !crop
      elseif (veg(i,j).eq.25) then    !warm irrigated crop
        pct_pft(i,j,0) = 15._r8       !includes bare ground and
        pct_pft(i,j,15) = 85._r8      !crop
      elseif (veg(i,j).eq.26) then    !warm crop
        pct_pft(i,j,0) = 15._r8       !includes bare ground and
        pct_pft(i,j,15) = 85._r8      !crop
      elseif (veg(i,j).eq.27) then    !forest wetland
        pct_pft(i,j,0) = 20._r8       !includes bare ground and
        pct_pft(i,j,5) = 80._r8       !broadleaf evergreen temperate tree
      elseif (veg(i,j).eq.28) then    !non-forest wetland
        pct_pft(i,j,0) = 100._r8      !is bare
      elseif (veg(i,j) < 0 .or. veg(i,j) > 28) then
        write(*,*) 'ERROR mapping veg to pct_pft: veg < 0 OR veg > 28',veg(i,j),i,j
      end if
    end do
  end do

! error check
  sumpctpft = sum(pct_pft, dim=3) !sum of %pft for every grid cell
  do j = 1, nlat
    do i = 1, nlon
      if (landmask(i,j)==1 .and. sumpctpft(i,j)/=100.) then
        write(*,*) 'ERROR: sumpctpft =',sumpctpft(i,j),i,j
      end if
    end do
  end do

! -------------------------------------------------------------------------- 
! Write variables
! --------------------------------------------------------------------------

  call wrap_put_var_realx (ncid, lon_id     , lon)
  call wrap_put_var_realx (ncid, lat_id     , lat)
  call wrap_put_var_realx (ncid, longxy_id  , longxy)
  call wrap_put_var_realx (ncid, latixy_id  , latixy)
  call wrap_put_var_realx (ncid, edgen_id   , edge(1))
  call wrap_put_var_realx (ncid, edgee_id   , edge(2))
  call wrap_put_var_realx (ncid, edges_id   , edge(3))
  call wrap_put_var_realx (ncid, edgew_id   , edge(4))
  call wrap_put_var_realx (ncid, landmask_id, landmask)
  call wrap_put_var_realx (ncid, pct_pft_id , pct_pft)

  call wrap_close(ncid)

end subroutine create_mksrf_pft



!============================================================================
subroutine create_mksrf_lai(veg,landmask,numpft,		  &
			    nlon,nlat,lon,longxy,lat,latixy,edge, &
                            fileo,ncid,                           &
                            pct_pft) 


  implicit none
  include 'netcdf.inc'

! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid


  integer :: nlon,nlat,numpft,ncid       !number lats/lons/pfts, nc fileo id 
  character(len=80) :: fileo             !output filenames

  real(r8) :: pct_pft(nlon,nlat,0:numpft) !percent pft 

! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------

  real(r8) mlai (nlon,nlat,0:numpft) !monthly lai in
  real(r8) msai (nlon,nlat,0:numpft) !monthly sai in
  real(r8) mhgtt(nlon,nlat,0:numpft) !monthly height (top) in
  real(r8) mhgtb(nlon,nlat,0:numpft) !monthly height (bottom) in

  integer :: dimlon_id               !netCDF dimension id
  integer :: dimlat_id               !netCDF dimension id
  integer :: dimpft_id               !netCDF dimension id
  integer :: dimtim_id               !netCDF dimension id

  integer :: lon_id                  !1d longitude array id
  integer :: lat_id                  !1d latitude array id
  integer :: longxy_id               !2d longitude array id
  integer :: latixy_id               !2d latitude array id
  integer :: edgen_id                !northern edge of grid (edge(1)) id
  integer :: edgee_id                !eastern  edge of grid (edge(2)) id
  integer :: edges_id                !southern edge of grid (edge(3)) id
  integer :: edgew_id                !western  edge of grid (edge(4)) id
  integer :: landmask_id             !landmask id
  integer :: mlai_id                 !monthly mlai id
  integer :: msai_id                 !monthly msai id  
  integer :: mhgtt_id                !monthly mghtt id
  integer :: mhgtb_id                !monthly mhgtb id 
 
  integer :: ntim                    !month time index
  integer :: i,j,l                   !indices
  integer :: beg4d(4),len4d(4)       !netCDF edge
  integer :: dim1_id(1)              !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)              !netCDF dimension id for 2-d variables
  integer :: dim4_id(4)              !netCDF dimension id for 4-d variables
  integer :: status                  !status

  character(len=80) :: name,unit     !netCDF attributes

  integer :: month
  real(r8) gai(14,12)      !leaf area index (one sided)      |taken from LSM's
  real(r8) tai(14,12)      !leaf+stem area index (one sided) |subroutine
  real(r8) hvt(14)         !top of canopy (m)                |vegconi.F
  real(r8) hvb(14)         !bottom of canopy (m)             

  data (tai(1,i),i=1,12)  /4.5,4.7,5.0,5.1,5.3,5.5,5.3,5.3,5.2,4.9,4.6,4.5/
  data (tai(2,i),i=1,12)  /0.3,0.3,0.3,1.0,1.6,2.4,4.3,2.9,2.0,1.3,0.8,0.5/
  data (tai(3,i),i=1,12)  /5.0,5.0,5.0,5.0,5.0,5.0,5.0,5.0,5.0,5.0,5.0,5.0/
  data (tai(4,i),i=1,12)  /0.4,0.4,0.7,1.6,3.5,5.1,5.4,4.8,3.8,1.7,0.6,0.4/
  data (tai(5,i),i=1,12)  /1.2,1.0,0.9,0.8,0.8,1.0,2.0,3.7,3.2,2.7,1.9,1.2/
  data (tai(6,i),i=1,12)  /0.7,0.8,0.9,1.0,1.5,3.4,4.3,3.8,1.8,1.0,0.9,0.8/
  data (tai(7,i),i=1,12)  /1.3,1.3,1.3,1.3,1.3,1.3,1.3,1.3,1.3,1.3,1.3,1.3/
  data (tai(8,i),i=1,12)  /1.0,1.0,0.8,0.3,0.6,0.0,0.1,0.3,0.5,0.6,0.7,0.9/
  data (tai(9,i),i=1,12)  /0.1,0.1,0.1,0.1,0.1,0.3,1.5,1.7,1.4,0.1,0.1,0.1/
  data (tai(10,i),i=1,12) /0.7,0.8,0.9,1.0,1.5,3.4,4.3,3.8,1.8,1.0,0.9,0.8/
  data (tai(11,i),i=1,12) /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
  data (tai(12,i),i=1,12) /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
  data (tai(13,i),i=1,12) /0.7,0.8,0.9,1.0,1.5,3.4,4.3,3.8,1.8,1.0,0.9,0.8/
  data (tai(14,i),i=1,12) /0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0/

  data (gai(1,i),i=1,12)  /4.1,4.2,4.6,4.8,4.9,5.0,4.8,4.7,4.6,4.2,4.0,4.0/
  data (gai(2,i),i=1,12)  /0.0,0.0,0.0,0.6,1.2,2.0,2.6,1.7,1.0,0.5,0.2,0.0/
  data (gai(3,i),i=1,12)  /4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5,4.5/
  data (gai(4,i),i=1,12)  /0.0,0.0,0.3,1.2,3.0,4.7,4.5,3.4,1.2,0.3,0.0,0.0/
  data (gai(5,i),i=1,12)  /0.8,0.7,0.4,0.5,0.5,0.7,1.7,3.0,2.5,1.6,1.0,1.0/
  data (gai(6,i),i=1,12)  /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
  data (gai(7,i),i=1,12)  /1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0,1.0/
  data (gai(8,i),i=1,12)  /0.9,0.8,0.2,0.2,0.0,0.0,0.0,0.2,0.4,0.5,0.6,0.8/
  data (gai(9,i),i=1,12)  /0.0,0.0,0.0,0.0,0.0,0.2,1.4,1.2,0.0,0.0,0.0,0.0/
  data (gai(10,i),i=1,12) /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
  data (gai(11,i),i=1,12) /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
  data (gai(12,i),i=1,12) /0.0,0.0,0.0,0.0,1.0,2.0,3.0,3.0,1.5,0.0,0.0,0.0/
  data (gai(13,i),i=1,12) /0.4,0.5,0.6,0.7,1.2,3.0,3.5,1.5,0.7,0.6,0.5,0.4/
  data (gai(14,i),i=1,12) /0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0/

  data hvt /17.0,14.0,35.0,20.0,18.0, 0.5, 0.5, &
             0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.0/

  data hvb / 8.50, 7.00, 1.00,11.50,10.00, 0.01, 0.10, &
             0.10, 0.10, 0.01, 0.01, 0.01, 0.01, 0.00/

! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes
 
  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'lai_sai_data')
  
! Define dimensions
  
  call wrap_def_dim (ncid, 'lon' , nlon        , dimlon_id)
  call wrap_def_dim (ncid, 'lat' , nlat        , dimlat_id)
  call wrap_def_dim (ncid, 'pft' , numpft+1    , dimpft_id)
  call wrap_def_dim (ncid, 'time', nf_unlimited, dimtim_id)
  
! Define input file independent variables 
  
  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)
  
  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)
  
  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)
  
  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)
  
  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)
  
  name = 'western edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)
     
! Define input file specific variables
  
  name = 'land mask'
  unit = 'unitless'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LANDMASK', nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

  dim4_id(1) = dimlon_id
  dim4_id(2) = dimlat_id
  dim4_id(3) = dimpft_id
  dim4_id(4) = dimtim_id

  name = 'monthly leaf area index'
  unit = 'unitless'
  call wrap_def_var (ncid ,'MONTHLY_LAI', nf_float, 4, dim4_id, mlai_id)
  call wrap_put_att_text (ncid, mlai_id, 'long_name', name)
  call wrap_put_att_text (ncid, mlai_id, 'units'    , unit)
     
  name = 'monthly stem area index'
  unit = 'unitless'
  call wrap_def_var (ncid ,'MONTHLY_SAI', nf_float, 4, dim4_id, msai_id)
  call wrap_put_att_text (ncid, msai_id, 'long_name', name)
  call wrap_put_att_text (ncid, msai_id, 'units'    , unit)
     
  name = 'monthly height top'
  unit = 'meters'
  call wrap_def_var (ncid ,'MONTHLY_HEIGHT_TOP', nf_float, 4, dim4_id, mhgtt_id)
  call wrap_put_att_text (ncid, mhgtt_id, 'long_name', name)
  call wrap_put_att_text (ncid, mhgtt_id, 'units'    , unit)
  
  name = 'monthly height bottom'
  unit = 'meters'
  call wrap_def_var (ncid ,'MONTHLY_HEIGHT_BOT', nf_float, 4, dim4_id, mhgtb_id)
  call wrap_put_att_text (ncid, mhgtb_id, 'long_name', name)
  call wrap_put_att_text (ncid, mhgtb_id, 'units'    , unit)
     
  status = nf_enddef(ncid)

! -----------------------------------------------------------------------
! Create monthly_lai, monthly_sai, monthly_height_top, monthly_height_bot
! from LSM vegtypes, and LSM annual cycle lookup table. LSM subroutine
! vegconi.F contains lookup tables for the variables gai, tai, hvt, hvb:
! 
! dimensioned 14 LSM pfts by 12 months
! ------------------------------------
! gai is LAI
! tai is LAI+SAI, so SAI=tai-gai
! 
! dimensioned 14 LSM pfts (constant in time)
! ------------------------------------------
! hvt: height at top of canopy
! hvb: height at bottom of canopy 
! -----------------------------------------------------------------------

  mlai = 0. !initialize four variables globally
  msai = 0. !these values will be used for ocean and for bare ground
  mhgtt = 0.
  mhgtb = 0.

! --------------------------------------------------------------------------
! Write variables
! --------------------------------------------------------------------------

  call wrap_put_var_realx (ncid, lon_id     , lon)
  call wrap_put_var_realx (ncid, lat_id     , lat)
  call wrap_put_var_realx (ncid, longxy_id  , longxy)
  call wrap_put_var_realx (ncid, latixy_id  , latixy)
  call wrap_put_var_realx (ncid, edgen_id   , edge(1))
  call wrap_put_var_realx (ncid, edgee_id   , edge(2))
  call wrap_put_var_realx (ncid, edges_id   , edge(3))
  call wrap_put_var_realx (ncid, edgew_id   , edge(4))


! write out landmask

  call wrap_put_var_realx (ncid, landmask_id, landmask)

! now enter time loop

  do ntim = 1,12

  do j = 1,nlat !this loop written by slevis
    if (j <= nlat/2 .and. ntim <= 6) then
      month = ntim + 6
    else if (j <= nlat/2 .and. ntim > 6) then
      month = ntim - 6
    else
      month = ntim
    end if
    do i = 1, nlon
      if (pct_pft(i,j,1) > 0) then
        mlai(i,j,1) = gai(1,month)
        msai(i,j,1) = tai(1,month)-gai(1,month)
        mhgtt(i,j,1) = hvt(1)
        mhgtb(i,j,1) = hvb(1)
      end if
      if (pct_pft(i,j,2) > 0) then
        mlai(i,j,2) = gai(1,month)
        msai(i,j,2) = tai(1,month)-gai(1,month)
        mhgtt(i,j,2) = hvt(1)
        mhgtb(i,j,2) = hvb(1)
      end if
      if (pct_pft(i,j,3) > 0) then
        mlai(i,j,3) = gai(2,month)
        msai(i,j,3) = tai(2,month)-gai(2,month)
        mhgtt(i,j,3) = hvt(2)
        mhgtb(i,j,3) = hvb(2)
      end if
      if (pct_pft(i,j,4) > 0) then
        mlai(i,j,4) = gai(3,month)
        msai(i,j,4) = tai(3,month)-gai(3,month)
        mhgtt(i,j,4) = hvt(3)
        mhgtb(i,j,4) = hvb(3)
      end if
      if (pct_pft(i,j,5) > 0) then
        mlai(i,j,5) = gai(3,month)
        msai(i,j,5) = tai(3,month)-gai(3,month)
        mhgtt(i,j,5) = hvt(3)
        mhgtb(i,j,5) = hvb(3)
      end if
      if (pct_pft(i,j,6) > 0) then
        mlai(i,j,6) = gai(5,month)
        msai(i,j,6) = tai(5,month)-gai(5,month)
        mhgtt(i,j,6) = hvt(5)
        mhgtb(i,j,6) = hvb(5)
      end if
      if (pct_pft(i,j,7) > 0) then
        mlai(i,j,7) = gai(4,month)
        msai(i,j,7) = tai(4,month)-gai(4,month)
        mhgtt(i,j,7) = hvt(4)
        mhgtb(i,j,7) = hvb(4)
      end if
      if (pct_pft(i,j,8) > 0) then
        mlai(i,j,8) = gai(4,month)
        msai(i,j,8) = tai(4,month)-gai(4,month)
        mhgtt(i,j,8) = hvt(4)
        mhgtb(i,j,8) = hvb(4)
      end if
      if (pct_pft(i,j,9) > 0) then
        mlai(i,j,9) = gai(7,month)
        msai(i,j,9) = tai(7,month)-gai(7,month)
        mhgtt(i,j,9) = hvt(7)
        mhgtb(i,j,9) = hvb(7)
      end if
      if (pct_pft(i,j,10) > 0) then
        mlai(i,j,10) = gai(8,month)
        msai(i,j,10) = tai(8,month)-gai(8,month)
        mhgtt(i,j,10) = hvt(8)
        mhgtb(i,j,10) = hvb(8)
      end if
      if (pct_pft(i,j,11) > 0) then
        mlai(i,j,11) = gai(9,month)
        msai(i,j,11) = tai(9,month)-gai(9,month)
        mhgtt(i,j,11) = hvt(9)
        mhgtb(i,j,11) = hvb(9)
      end if
      if (pct_pft(i,j,12) > 0) then
        mlai(i,j,12) = gai(10,month)
        msai(i,j,12) = tai(10,month)-gai(10,month)
        mhgtt(i,j,12) = hvt(10)
        mhgtb(i,j,12) = hvb(10)
      end if
      if (pct_pft(i,j,13) > 0) then
        mlai(i,j,13) = gai(6,month)
        msai(i,j,13) = tai(6,month)-gai(6,month)
        mhgtt(i,j,13) = hvt(6)
        mhgtb(i,j,13) = hvb(6)
      end if
      if (pct_pft(i,j,14) > 0) then
        mlai(i,j,14) = gai(13,month)
        msai(i,j,14) = tai(13,month)-gai(13,month)
        mhgtt(i,j,14) = hvt(13)
        mhgtb(i,j,14) = hvb(13)
      end if
      if (pct_pft(i,j,15) > 0) then
        mlai(i,j,15) = gai(11,month)
        msai(i,j,15) = tai(11,month)-gai(11,month)
        mhgtt(i,j,15) = hvt(11)
        mhgtb(i,j,15) = hvb(11)
      end if
    end do
  end do !end slevis loops

  beg4d(1) = 1     ; len4d(1) = nlon
  beg4d(2) = 1     ; len4d(2) = nlat
  beg4d(3) = 1     ; len4d(3) = numpft+1
  beg4d(4) = ntim  ; len4d(4) = 1
     
  call wrap_put_vara_realx (ncid, mlai_id , beg4d, len4d, mlai )
  call wrap_put_vara_realx (ncid, msai_id , beg4d, len4d, msai )
  call wrap_put_vara_realx (ncid, mhgtt_id, beg4d, len4d, mhgtt)
  call wrap_put_vara_realx (ncid, mhgtb_id, beg4d, len4d, mhgtb)

  end do   

  ! Close output file

  call wrap_close(ncid)

end subroutine create_mksrf_lai


!=================================================================================
subroutine create_mksrf_soicol(veg,landmask,			     &
			       nlon,nlat,lon,longxy,lat,latixy,edge, &
                               fileo,ncid)


  implicit none
  include 'netcdf.inc'

! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid

  integer :: nlon,nlat,ncid              !number lats/lons, nc fileo id 
  character(len=80) :: fileo             !output filenames


! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------

  real(r8) :: soil_color(nlon,nlat)       !lsm soil color

  integer :: dimlon_id                    !netCDF dimension id
  integer :: dimlat_id                    !netCDF dimension id

  integer :: lon_id                       !1d longitude array id
  integer :: lat_id                       !1d latitude array id
  integer :: longxy_id                    !2d longitude array id
  integer :: latixy_id                    !2d latitude array id
  integer :: edgen_id                     !northern edge of grid (edge(1)) id
  integer :: edgee_id                     !eastern  edge of grid (edge(2)) id
  integer :: edges_id                     !southern edge of grid (edge(3)) id
  integer :: edgew_id                     !western  edge of grid (edge(4)) id
  integer :: soil_color_id                !soil color id
  integer :: landmask_id                  !landmask id

  integer :: i,j                          !indices
  integer :: dim1_id(1)                   !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)                   !netCDF dimension id for 2-d variables
  integer :: status                       !status

  character(len=256) :: name,unit         !netCDF attributes


! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes

  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'soil_color_data')

! Define dimensions

  call wrap_def_dim (ncid, 'lon' , nlon, dimlon_id)
  call wrap_def_dim (ncid, 'lat' , nlat, dimlat_id)

! Define input file independent variables 

  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)

  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)

  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)

  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)

  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)

  name = 'western edge of surface grid'
  
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)

! Define input file specific variables

  name = 'soil color'
  unit = 'unitless'
  dim2_id(1) = lon_id
  dim2_id(2) = lat_id
  call wrap_def_var (ncid ,'SOIL_COLOR' ,nf_float, 2, dim2_id, soil_color_id)
  call wrap_put_att_text (ncid, soil_color_id, 'long_name', name)
  call wrap_put_att_text (ncid, soil_color_id, 'units'    , unit)

  name = 'land mask'
  unit = 'unitless'
  call wrap_def_var (ncid ,'LANDMASK' ,nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

! End of definition

  status = nf_enddef(ncid)

! -----------------------------------------------------------------------
! Create soil_color from LSM vegtypes (if user so desires)
! -----------------------------------------------------------------------

  do j = 1,nlon
  do i = 1,nlat
   if(landmask(j,i) == 1.) soil_color(j,i) = 4._r8
  enddo
  enddo
  print *, 'Soil Color sample at point (1,1)',soil_color(1,1)

! --------------------------------------------------------------------------
! Write variables
! --------------------------------------------------------------------------

  call wrap_put_var_realx (ncid, lon_id       , lon)
  call wrap_put_var_realx (ncid, lat_id       , lat)
  call wrap_put_var_realx (ncid, longxy_id    , longxy)
  call wrap_put_var_realx (ncid, latixy_id    , latixy)
  call wrap_put_var_realx (ncid, edgen_id     , edge(1))
  call wrap_put_var_realx (ncid, edgee_id     , edge(2))
  call wrap_put_var_realx (ncid, edges_id     , edge(3))
  call wrap_put_var_realx (ncid, edgew_id     , edge(4))
  call wrap_put_var_realx (ncid, landmask_id  , landmask)
  call wrap_put_var_realx (ncid, soil_color_id, soil_color)

! Close output file

  call wrap_close(ncid)

end subroutine create_mksrf_soicol



!===============================================================================
subroutine create_mksrf_soitex(veg,landmask,     		         &
			       dzsoi,zsoi,mapunits0,pct_clay0,pct_sand0, &
		               nlay,nlon_st,nlat_st,nmapunits,mapunitmax,&
		               nlon,nlat,lon,longxy,lat,latixy,edge,     &
                               fileo,ncid)


  implicit none
  include 'netcdf.inc'

! ---------------------------------------------------------------------
! Global variables
!-----------------------------------------------------------------------

  integer, parameter :: r8 = selected_real_kind(12)

  integer  :: veg(nlon,nlat)	          !lsm veg array  (input)
  real(r8) :: landmask(nlon,nlat)	  !landmask array  (input)
  real(r8) :: lon(nlon)                   !longitude dimension array (1d)
  real(r8) :: lat(nlat)                   !latitude dimension array (1d) 
  real(r8) :: longxy(nlon,nlat)           !longitude dimension array (2d)  
  real(r8) :: latixy(nlon,nlat)           !longitude dimension array (2d)
  real(r8) :: edge(4)                     !N,E,S,W edges of grid

  real(r8) :: dzsoi(10), zsoi(10)        !soil layer thickness and depth
  real(r8) :: pct_sand0(nlay,mapunitmax)    !original percent sand 
  real(r8) :: pct_clay0(nlay,mapunitmax)    !original percent clay 
  real(r8) :: mapunits0(nlon_st,nlat_st)    !mapunits 
  

  integer :: nlon_st,nlat_st,nlay	 !original soil tex dims
  integer :: nmapunits,mapunitmax        !# igbp mapunits, max value mu 
  integer :: nlon,nlat,ncid 		 !number lats/lons, nc fileo id 
  character(len=80) :: fileo             !output filenames


! ------------------------------------------------------------------
! Define local variables
! ------------------------------------------------------------------


  integer  :: countloam                   !layers of loam per mapunit
  integer  :: mu                          !current mapunit
  real(r8) :: pct_sand(mapunitmax,nlay)   !pct sand 
  real(r8) :: pct_clay(mapunitmax,nlay)   !pct clay
  real(r8) :: mapunits(nlon,nlat)         !new mapunits

  integer :: dimlon_id                    !netCDF dimension id
  integer :: dimlat_id                    !netCDF dimension id
  integer :: dimlay_id                    !netCDF dimension id
  integer :: dimmapunits_id               !netCDF dimension id
  integer :: dimmapunitmax_id             !netCDF dimension id

  integer :: dzsoi_id                     !soil thickness by layer
  integer :: zsoi_id                      !soil depth by layer
  integer :: lon_id                       !1d longitude array id
  integer :: lat_id                       !1d latitude array id
  integer :: lay_id                       !1d layer array id
  integer :: mapunit_id                   !2d mapunits array id
  integer :: longxy_id                    !2d longitude array id
  integer :: latixy_id                    !2d latitude array id
  integer :: edgen_id                     !northern edge of grid (edge(1)) id
  integer :: edgee_id                     !eastern  edge of grid (edge(2)) id
  integer :: edges_id                     !southern edge of grid (edge(3)) id
  integer :: edgew_id                     !western  edge of grid (edge(4)) id
  integer :: pct_sand_id                  !sand id
  integer :: pct_clay_id                  !clay id
  integer :: landmask_id                  !landmask id

  integer :: i,j,k                        !indices
  integer :: dim1_id(1)                   !netCDF dimension id for 1-d variables
  integer :: dim2_id(2)                   !netCDF dimension id for 2-d variables
  integer :: status                       !status

  character(len=256) :: name,unit         !netCDF attributes


! ------------------------------------------------------------
! Create skeleton netcdf
! --------------------------------------------------------------

! Define global attributes

  call wrap_create (fileo, nf_clobber, ncid)
  call wrap_put_att_text (ncid, nf_global, 'data_type', 'igbp_soil_texture_data')

! Define dimensions

  call wrap_def_dim (ncid, 'lon' , nlon, dimlon_id)
  call wrap_def_dim (ncid, 'lat' , nlat, dimlat_id)
  call wrap_def_dim (ncid, 'number_of_layers'   , nlay      , dimlay_id)
  call wrap_def_dim (ncid, 'number_of_mapunits' , nmapunits , dimmapunits_id)
  call wrap_def_dim (ncid, 'max_value_mapunit'  , mapunitmax, dimmapunitmax_id)

! Define input file independent variables 

  name = 'lon'
  unit = 'degrees east'
  dim1_id(1) = dimlon_id
  call wrap_def_var (ncid,'lon', nf_float, 1, dim1_id, lon_id)
  call wrap_put_att_text (ncid, lon_id, 'long_name', name)
  call wrap_put_att_text (ncid, lon_id, 'units'    , unit)

  name = 'lat'
  unit = 'degrees north'
  dim1_id(1) = dimlat_id
  call wrap_def_var (ncid,'lat', nf_float, 1, dim1_id, lat_id)
  call wrap_put_att_text (ncid, lat_id, 'long_name', name)
  call wrap_put_att_text (ncid, lat_id, 'units'    , unit)

  name = 'longitude-2d'
  unit = 'degrees east'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid, 'LONGXY', nf_float, 2, dim2_id, longxy_id)
  call wrap_put_att_text (ncid, longxy_id, 'long_name', name)
  call wrap_put_att_text (ncid, longxy_id, 'units'    , unit)

  name = 'latitude-2d'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'LATIXY', nf_float, 2, dim2_id, latixy_id)
  call wrap_put_att_text (ncid, latixy_id, 'long_name', name)
  call wrap_put_att_text (ncid, latixy_id, 'units'    , unit)

  name = 'land mask'
  unit = 'unitless'
  call wrap_def_var (ncid ,'LANDMASK' ,nf_float, 2, dim2_id, landmask_id)
  call wrap_put_att_text (ncid, landmask_id, 'long_name', name)
  call wrap_put_att_text (ncid, landmask_id, 'units'    , unit)

! to possibly replace the next two variables
! find out about dimensioned variables
! (eg, see how pressure levels are treated)

  name = 'soil layer thickness'
  unit = 'm'
  dim1_id(1) = dimlay_id
  call wrap_def_var (ncid,'DZSOI', nf_float, 1, dim1_id, dzsoi_id)
  call wrap_put_att_text (ncid, dzsoi_id, 'long_name', name)
  call wrap_put_att_text (ncid, dzsoi_id, 'units'    , unit)

  name = 'soil layer depth'
  unit = 'm'
  dim1_id(1) = dimlay_id
  call wrap_def_var (ncid,'ZSOI', nf_float, 1, dim1_id, zsoi_id)
  call wrap_put_att_text (ncid, zsoi_id, 'long_name', name)
  call wrap_put_att_text (ncid, zsoi_id, 'units'    , unit)

  name = 'northern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGEN', nf_float, 0, 0, edgen_id)
  call wrap_put_att_text (ncid, edgen_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgen_id, 'units'    , unit)

  name = 'eastern edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEE', nf_float, 0, 0, edgee_id)
  call wrap_put_att_text (ncid, edgee_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgee_id, 'units'    , unit)

  name = 'southern edge of surface grid'
  unit = 'degrees north'
  call wrap_def_var (ncid, 'EDGES', nf_float, 0, 0, edges_id)
  call wrap_put_att_text (ncid, edges_id, 'long_name', name)
  call wrap_put_att_text (ncid, edges_id, 'units'    , unit)

  name = 'western edge of surface grid'
  unit = 'degrees east'
  call wrap_def_var (ncid, 'EDGEW', nf_float, 0, 0, edgew_id)
  call wrap_put_att_text (ncid, edgew_id, 'long_name', name)
  call wrap_put_att_text (ncid, edgew_id, 'units'    , unit)

! Define soil type and soil texture variables

  name = 'igbp soil mapunit'
  unit = 'unitless'
  dim2_id(1) = dimlon_id
  dim2_id(2) = dimlat_id
  call wrap_def_var (ncid ,'MAPUNITS' ,nf_float, 2, dim2_id, mapunit_id)
  call wrap_put_att_text (ncid, mapunit_id, 'long_name', name)
  call wrap_put_att_text (ncid, mapunit_id, 'units'    , unit)

  name = 'percent sand'
  unit = 'unitless'
  dim2_id(1) = dimmapunitmax_id
  dim2_id(2) = dimlay_id
  call wrap_def_var (ncid ,'PCT_SAND' ,nf_float, 2, dim2_id, pct_sand_id)
  call wrap_put_att_text (ncid, pct_sand_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_sand_id, 'units'    , unit)

  name = 'percent clay'
  unit = 'unitless'
  call wrap_def_var (ncid ,'PCT_CLAY' ,nf_float, 2, dim2_id, pct_clay_id)
  call wrap_put_att_text (ncid, pct_clay_id, 'long_name', name)
  call wrap_put_att_text (ncid, pct_clay_id, 'units'    , unit)

! End of definition

  status = nf_enddef(ncid)


! -----------------------------------------------------------------------
! pct_clay and pct_sand data 
! Think of mapunits as soil profiles. Each profile corresponds to a unique
! set of %sand and %clay values from the surface layer to the deepest one.
!
! Here we look for the mapunit that corresponds to a loam (an average soil)
! for all layers. We assign that mapunit to all land points.
! Users with more extensive knowledge of the soil texture for the period
! of their interest, should expand this code accordingly.
! -----------------------------------------------------------------------

  pct_sand = 0.  !initialize
  pct_clay = 0.  !three
  mapunits = 0.  !output variables

  do mu = 1,mapunitmax
    countloam = 0
    do k = 1,nlay
      if (pct_sand0(k,mu) > 39 .and. pct_sand0(k,mu) < 47 .and. &
          pct_clay0(k,mu) > 14 .and. pct_clay0(k,mu) < 22) then
        countloam = countloam + 1
      end if
      if (countloam == 10) then  !all layers are loam
        write(*,*) 'Found loam to be mapunit:',mu
        do j = 1,nlat
          do i = 1,nlon
            if (landmask(i,j) == 1) then
              mapunits(i,j) = mu !set mapunits globally to current mu (loam)
            end if
          end do !lon loop
        end do   !lat loop
      end if
      pct_sand(mu,k) = pct_sand0(k,mu) !the mapping from mapunit to %sand
      pct_clay(mu,k) = pct_clay0(k,mu) !and %clay remains the same
    end do       !loop through soil layers
  end do         !loop through mapunits

! --------------------------------------------------------------------------
! Write variables
! --------------------------------------------------------------------------

  call wrap_put_var_realx (ncid, lon_id     , lon)
  call wrap_put_var_realx (ncid, lat_id     , lat)
  call wrap_put_var_realx (ncid, longxy_id  , longxy)
  call wrap_put_var_realx (ncid, latixy_id  , latixy)
  call wrap_put_var_realx (ncid, landmask_id, landmask)
  call wrap_put_var_realx (ncid, edgen_id   , edge(1))
  call wrap_put_var_realx (ncid, edgee_id   , edge(2))
  call wrap_put_var_realx (ncid, edges_id   , edge(3))
  call wrap_put_var_realx (ncid, edgew_id   , edge(4))
  call wrap_put_var_realx (ncid, dzsoi_id   , dzsoi)
  call wrap_put_var_realx (ncid, zsoi_id    , zsoi)
  call wrap_put_var_realx (ncid, mapunit_id , mapunits)
  call wrap_put_var_realx (ncid, pct_sand_id, pct_sand)
  call wrap_put_var_realx (ncid, pct_clay_id, pct_clay)

  call wrap_close(ncid)

end subroutine create_mksrf_soitex 


!================================================================================
!================================================================================

subroutine wrap_create (path, cmode, ncid)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  character(len=*) path
  integer cmode, ncid, ret
  ret = nf_create (path, cmode, ncid)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_create

!===============================================================================

subroutine wrap_def_dim (nfid, dimname, len, dimid)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: nfid, len, dimid
  character(len=*) :: dimname
  integer ret
  ret = nf_def_dim (nfid, dimname, len, dimid)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_def_dim

!===============================================================================

subroutine wrap_def_var (nfid, name, xtype, nvdims, vdims, varid)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: nfid, xtype, nvdims, varid
  integer :: vdims(nvdims)
  character(len=*) :: name
  integer ret
  ret = nf_def_var (nfid, name, xtype, nvdims, vdims, varid)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_def_var

!===============================================================================

subroutine wrap_put_att_text (nfid, varid, attname, atttext)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: nfid, varid
  character(len=*) :: attname, atttext
  integer :: ret, siz
  siz = len_trim(atttext)
  ret = nf_put_att_text (nfid, varid, attname, siz, atttext)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_put_att_text

!===============================================================================

subroutine wrap_put_var_realx (nfid, varid, arr)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: nfid, varid
  real(r8) :: arr(*)
  integer :: ret
#ifdef CRAY
  ret = nf_put_var_real (nfid, varid, arr)
#else
  ret = nf_put_var_double (nfid, varid, arr)
#endif
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_put_var_realx

!===============================================================================

subroutine wrap_put_var_int (nfid, varid, arr)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: nfid, varid
  integer :: arr(*)
  integer :: ret
  ret = nf_put_var_int (nfid, varid, arr)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_put_var_int

!===============================================================================

subroutine wrap_put_vara_realx (nfid, varid, start, count, arr)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: nfid, varid
  integer :: start(*), count(*)
  real(r8) arr(*)
  integer ret
#ifdef CRAY
  ret = nf_put_vara_real (nfid, varid, start, count, arr)
#else
  ret = nf_put_vara_double (nfid, varid, start, count, arr)
#endif
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_put_vara_realx

  
!===============================================================================

subroutine wrap_close (ncid)
  implicit none
  include 'netcdf.inc'
  integer, parameter :: r8 = selected_real_kind(12)
  integer :: ncid
  integer :: ret
  ret = nf_close (ncid)
  if (ret.ne.NF_NOERR) then
     write(6,*)'WRAP_CLOSE: nf_close failed for id ',ncid
     call handle_error (ret)
  end if
end subroutine wrap_close

!===============================================================================

subroutine handle_error(ret)
  implicit none
  include 'netcdf.inc'
  integer :: ret
  if (ret .ne. nf_noerr) then
     write(6,*) 'NCDERR: ERROR: ',nf_strerror(ret)
     call abort
  endif
end subroutine handle_error

!==============================================================================

subroutine wrap_inq_varid (nfid, varname, varid)
  implicit none
  include 'netcdf.inc'

  integer nfid, varid
  character*(*) varname

  integer ret

  ret = nf_inq_varid (nfid, varname, varid)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_inq_varid

!=============================================================================

subroutine wrap_get_var8 (nfid, varid, arr)
  implicit none
  include 'netcdf.inc'

  integer nfid, varid
  real*8 arr(*)

  integer ret

  ret = nf_get_var_double (nfid, varid, arr)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_get_var8

!=============================================================================

subroutine wrap_get_int (nfid, varid, arr)
  implicit none
  include 'netcdf.inc'

  integer nfid, varid
  integer arr(*)

  integer ret

  ret = nf_get_var_int (nfid, varid, arr)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_get_int

!=============================================================================


subroutine wrap_get_flt (nfid, varid, arr)
  implicit none
  include 'netcdf.inc'

  integer nfid, varid
  real arr(*)

  integer ret

  ret = nf_get_var_real (nfid, varid, arr)
  if (ret.ne.NF_NOERR) call handle_error (ret)
end subroutine wrap_get_flt

!==============================================================================

subroutine endrun
  implicit none
  include 'netcdf.inc'

  call abort
  stop 999
end subroutine endrun


!===============================================================================


