!===============================================================================
! SVN $Id: smooth_mod.F90 2701 2006-12-19 23:24:43Z kauff $
! SVN $URL: https://svn-ccsm-models.cgd.ucar.edu/tools/mapping/gen_runoffmap/trunk/src/smooth_mod.F90 $
!===============================================================================

MODULE smooth_mod

   use map_mod

   implicit none

!-----------------------------------------------------------------------
!
!     module variables
!
!-----------------------------------------------------------------------

   real,parameter :: rEarth   =  6.37122e6    ! radius of earth ~ m
   real,parameter :: DtoR     =  3.14159265358979323846/180.0  
   real, allocatable    :: garr(:,:,:)   ! dummy runoff (nx,ny,nbasin) in
					 ! kg/s/m^2

!===============================================================================
CONTAINS 
!===============================================================================

SUBROUTINE smooth_init(map_in, map_out)

   implicit none

   !--- arguments ---
   type(sMatrix),intent( in)   :: map_in
   type(sMatrix),intent(inout) :: map_out

   !--- local ---
   integer :: rcode ! return code
   integer :: ns    ! number of links
   integer :: nactive

!-------------------------------------------------------------------------------
! PURPOSE:
! o Given map_in create map_out which maps from domain_b of
!   map_in to itself.  This is a template for the square smoothing 
!   map, which we assume is needed for domain_b of map_in.
! o Deallocation below copied from map_dup routine.  Purpose unclear.
!-------------------------------------------------------------------------------

   !------------------------------------------------
   ! de-allocate space
   !------------------------------------------------
   deallocate(map_out%  xc_a,STAT=rcode)
   deallocate(map_out%  yc_a,STAT=rcode)
   deallocate(map_out%  xv_a,STAT=rcode)
   deallocate(map_out%  yv_a,STAT=rcode)
   deallocate(map_out%mask_a,STAT=rcode)
   deallocate(map_out%area_a,STAT=rcode)

   deallocate(map_out%  xc_b,STAT=rcode)
   deallocate(map_out%  yc_b,STAT=rcode)
   deallocate(map_out%  xv_b,STAT=rcode)
   deallocate(map_out%  yv_b,STAT=rcode)
   deallocate(map_out%mask_b,STAT=rcode)
   deallocate(map_out%area_b,STAT=rcode)

   deallocate(map_out%frac_a,STAT=rcode)
   deallocate(map_out%frac_b,STAT=rcode)

   deallocate(map_out%s     ,STAT=rcode)
   deallocate(map_out%row   ,STAT=rcode)
   deallocate(map_out%col   ,STAT=rcode)
   deallocate(map_out%sn1   ,STAT=rcode)
   deallocate(map_out%sn2   ,STAT=rcode)

   !------------------------------------------------
   ! allocate space
   !------------------------------------------------
   nactive = count(map_in%mask_b > 0)
   ns = nactive*300	! assume max 300 dst points per src

   allocate(map_out%  xc_a(  map_in%n_b) )
   allocate(map_out%  yc_a(  map_in%n_b) )
   allocate(map_out%  xv_a(4,map_in%n_b) )
   allocate(map_out%  yv_a(4,map_in%n_b) )
   allocate(map_out%mask_a(  map_in%n_b) )
   allocate(map_out%area_a(  map_in%n_b) )

   allocate(map_out%  xc_b(  map_in%n_b) )
   allocate(map_out%  yc_b(  map_in%n_b) )
   allocate(map_out%  xv_b(4,map_in%n_b) )
   allocate(map_out%  yv_b(4,map_in%n_b) )
   allocate(map_out%mask_b(  map_in%n_b) )
   allocate(map_out%area_b(  map_in%n_b) )

   allocate(map_out%frac_a(map_in%n_b) )
   allocate(map_out%frac_b(map_in%n_b) )

   allocate(map_out%s(ns))
   allocate(map_out%row(ns))
   allocate(map_out%col(ns))
   allocate(map_out%sn1   (map_in%n_b) )
   allocate(map_out%sn2   (map_in%n_b) )

   !------------------------------------------------
   ! set values
   !------------------------------------------------
   map_out%   n_a = map_in%   n_b
   map_out%  ni_a = map_in%  ni_b
   map_out%  nj_a = map_in%  nj_b
   map_out%  xc_a = map_in%  xc_b
   map_out%  yc_a = map_in%  yc_b
   map_out%  xv_a = map_in%  xv_b
   map_out%  yv_a = map_in%  yv_b
   map_out%mask_a = map_in%mask_b
   map_out%area_a = map_in%area_b

   map_out%   n_b = map_in%   n_b
   map_out%  ni_b = map_in%  ni_b
   map_out%  nj_b = map_in%  nj_b
   map_out%  xc_b = map_in%  xc_b
   map_out%  yc_b = map_in%  yc_b
   map_out%  xv_b = map_in%  xv_b
   map_out%  yv_b = map_in%  yv_b
   map_out%mask_b = map_in%mask_b
   map_out%area_b = map_in%area_b

!  map_out%frac_a = map_in%frac_b
!  map_out%frac_b = map_in%frac_b
   map_out%frac_a = 1.0
   map_out%frac_b = 1.0

   map_out%n_s    = ns
   map_out%s      = 1.0
   map_out%row    = 1
   map_out%col    = 1
   map_out%sn1    = map_in%sn1
   map_out%sn2    = map_in%sn2

   map_out%title      = "Conservative Smoothing Map"
   map_out%normal     = map_in%normal
   map_out%method     = "created using fluxsmooth_mod.F90"
   map_out%history    = map_in%history
   map_out%convention = map_in%convention
   map_out%domain_a   = map_in%domain_b
   map_out%domain_b   = map_in%domain_b

END SUBROUTINE smooth_init

!===============================================================================

SUBROUTINE restrictsources(map,filename)

   implicit none

   !--- arguments ---
   type(sMatrix), intent(inout) :: map
   character(*) , intent(in)    :: filename  ! name of data file

   !--- local ---
   real,allocatable :: array_in(:,:)	     ! mask array read in
   real,allocatable :: array_in2(:)	     ! mask array read in

   !--- formats ---
   character(len=*),parameter :: F00 = "('(restrictsources) ',a,5i11)"

!-------------------------------------------------------------------------------
! PURPOSE:
! o Given 'map', a smoothing map that links ocean to ocean, redefine map%mask_a
!   based on the array read in from 'filename' in order to restrict the number
!   of source points that the smoothing routine has to deal with.
!   For example, if array_in is non-zero only for coastal points, then only
!   those points will be processed for the smoothing map, greatly reducing
!   the time of computation.
!
! NOTES:
! o Assumes that array read from 'filename' is dimensioned 
!   real(map%ni_a,map%nj_a)
! o Assumes array_in>0. means active, array_in<=0. mean inactive.
!-------------------------------------------------------------------------------

   allocate(array_in(map%ni_a,map%nj_a))
   allocate(array_in2(map%n_a))

   write(6,*)'(restrictsources)',map%ni_a
   write(6,*)'(restrictsources)',map%nj_a
   write(6,*)'(restrictsources)',filename
   write(6,*)'(restrictsources)',map%ni_a*map%nj_a*8
 
   open(10,file=filename,form='unformatted')
   read(10,err=1) array_in
1  write(6,F00) 'ERROR reading array_in'
   close(10)

   array_in2 = reshape(array_in,(/map%n_a/))

   map%mask_a = 0
   where(array_in2 > 0.0) map%mask_a = 1

END SUBROUTINE restrictsources

!===============================================================================

SUBROUTINE smooth(map,efold,rmax)

   implicit none

   !--- arguments ---
   type(sMatrix), intent(inout) :: map       
   real, intent(in) :: efold			! efold scale (m)
   real, intent(in) :: rmax			! max smoothing distance (m)

   !--- local ---
   integer         :: i,j,i0,i1,j0,j1,ic,jc,itmp,jtmp,k
   integer         :: ii,jj,ip,np,ni,nj,n_s,ngood,nbox
   integer         :: itest,jtest,k1,k2,k3,k4,gdcnt,cnt
   integer         :: mingd,maxgd,avggd,mingdi,mingdj,maxgdi,maxgdj
   integer         :: minbx,maxbx,minbxi,minbxj,maxbxi,maxbxj
   integer         :: onegd,nactivea,nactiveb,ntot
   integer, allocatable :: iind(:),jind(:),imask(:,:)
   real, allocatable :: rdist(:,:),areaa(:,:),wgt(:,:)
   real, allocatable :: s(:),row(:),col(:)
   real :: wgtsum
   real, parameter :: rmiss = 1.e30

   !--- formats ---
   character(len=*),parameter :: F1 = "('<smooth> ',a,2i11)"
   character(len=*),parameter :: F1a= "('<smooth> ',a,4i11)"
   character(len=*),parameter :: F2 = "('<smooth> ',a,2F11.6)"
   character(len=*),parameter :: F3 = "('<smooth> ',a,es18.7)"
   character(len=*),parameter :: F4 = "('<smooth> ',a,2i11,es18.7)"
   character(len=*),parameter :: F4a = "('<smooth> ',a,3i6,es18.7)"
   character(len=*),parameter :: F5 = "('<smooth> ',a,6i6)"
   character(len=*),parameter :: F6 = "('<smooth> ',a,3i6,es18.7,i6)"
   character(len=*),parameter :: F7 = "('<smooth> ',a,2i6,es18.7,3i6)"
   character(len=*),parameter :: F8 = "('<smooth> ',a,i6,F11.6)"
   character(len=*),parameter :: F9 = "('<smooth> ',a,i6,' @  (',i4,',',i4,')')"

!-------------------------------------------------------------------------------
! PURPOSE:
! o Given a square sMatrix, map, generate the links for local smoothing
!   of input fluxes with following properties:
!	-must be conservative
!	-must redistribute onto active points only 
!	-uses a 2D Gaussian defined by efold & rmax for weights
!	-weights are a function of shortest active distance, so that
!	 weights are zero across land isthmuses
!
!             -------------------------------------------------
!             |               |               |               |
!             |               |               |               |
!             |               |               |               |
!             |   ip=2        |    ip=3       |    ip=4       |
!    j1       |               |               |       ^       |
!             |               |               |       |       |
!             |               |               |       |d2     |
!             |               |               |       |       |
!             ----------------------------------------|--------
!             |               |               |       |       |
!             |               |               |       |       |
!             |   ip=1        |   (ic,jc)     |    (ii,jj)    |
!             |               |               |d1     |       |  d3
!             |               |    source<------------x-------|------->
!             |               |    cell       |       |       |
!             |               |               |    ip=5       |
!             |               |               |       |       |
!             ----------------------------------------|--------
!             |               |               |       |d4     |
!             |               |               |       |       |
!             |   ip=8        |    ip=7       |    ip=6       |
!    j0       |               |               |       |       |
!             |               |               |       v       |
!             |               |               |               |
!             |               |               |               |
!             |               |               |               |
!             -------------------------------------------------
!
!                    i0                              i1
!
!-------------------------------------------------------------------------------

   ni = map%ni_a
   nj = map%nj_a
   allocate(iind(map%n_a))
   allocate(jind(map%n_a))
   allocate(wgt(ni,nj))
   allocate(rdist(ni,nj))
   allocate(areaa(ni,nj))
   allocate(imask(ni,nj))
   do j=1,map%n_a
      iind(j) = mod(j-1,ni)+1
      jind(j) = int((j-1)/ni)+1
      imask(iind(j),jind(j)) = map%mask_b(j)
      areaa(iind(j),jind(j)) = map%area_a(j)
   enddo

   itest = 70
   jtest = 87

   n_s = 0
   minbx = 1000
   maxbx = 0
   mingd = 1000
   maxgd = 0
   avggd = 0
   onegd = 0
   write(6,F1) 'ni = ',ni
   write(6,F1) 'nj = ',nj
   write(6,F1) 'n_a = ',map%n_a
   write(6,F1) 'n_b = ',map%n_b
   write(6,F1) 'n_s = ',map%n_s
   write(6,F2) 'range(xc_a) = ',minval(map%xc_a(1:map%n_a)), &
  &	maxval(map%xc_a(1:map%n_a))
   write(6,F2) 'range(yc_a) = ',minval(map%yc_a(1:map%n_a)), &
  &	maxval(map%yc_a(1:map%n_a))
   write(6,F2) 'using efold (km) of = ',efold/1000.
   write(6,F2) 'using max radius (km) of = ',rmax/1000.

   !-----------------------------------------------
   ! loop over source points
   !-----------------------------------------------
   do j=1,map%n_a

 !   ! source points must have nonzero mask value
    if (map%mask_a(j) /= 0)  then

     ic = iind(j)
     jc = jind(j)

     rdist = 1.e20
     wgt = 0.0
     wgtsum = 0.0
     nbox = 0
     gdcnt = 0

     call recur_setDist(ic,jc,ni,nj,map%xc_a,map%yc_a,0.0,imask,rdist,rmax)

     where(rdist < 1.e20)
       wgt = exp(-rdist/efold)*areaa
     end where
     
     wgtsum = sum(wgt)

     !-----------------------------------------------
     ! loop over destination points
     !-----------------------------------------------
     do i=1,map%n_b
       ii = iind(i)
       jj = jind(i)
       if (wgt(ii,jj) > 0.0) then
	   n_s = n_s + 1
!          write(6,F1) 'n_s = ',n_s
           map%s(n_s) = (map%area_a(j)/map%area_b(i))*(wgt(ii,jj)/wgtsum)
           map%col(n_s) = j
           map%row(n_s) = i
       endif
     end do
   else                ! mask_a = 0 (not considered for smoothing,
			!    but if mask_b shows that it is an active
			!    ocean point, then fill in 1.0 on diagonal just
			!    to ensure conservation).
     if(map%mask_b(j) /= 0) then
       n_s = n_s + 1
       ii = iind(j)
       jj = jind(j)
       map%s(n_s) = 1.0
       map%col(n_s) = j
       map%row(n_s) = j
     endif
   endif		!  if mask_a /= 0
   enddo		!  loop over j=1,map%n_a

   avggd = int(avggd/count(map%mask_a == 1))

   write(6,F3) "done.       "
   write(6,F3) "min wgt     ",minval(wgt)
   write(6,F3) "max wgt     ",maxval(wgt)
   write(6,F3) "wgtsum      ",wgtsum
   write(6,F3) "min S       ",minval(map%s(1:n_s))
   write(6,F3) "max S       ",maxval(map%s(1:n_s))
   write(6,F1) "min row     ",minval(map%row(1:n_s))
   write(6,F1) "max row     ",maxval(map%row(1:n_s))
   write(6,F1) "min col     ",minval(map%col(1:n_s))
   write(6,F1) "max col     ",maxval(map%col(1:n_s))
   write(6,F9) "min # dest pts : ",mingd,mingdi,mingdj
   write(6,F9) "max # dest pts : ",maxgd,maxgdi,maxgdj
   write(6,F6) "avg # dest pts : ",avggd
   write(6,F6) "src # having only 1 dest  : ",onegd
   write(6,F9) "min # boxes    : ",minbx,minbxi,minbxj
   write(6,F9) "max # boxes    : ",maxbx,maxbxi,maxbxj
   write(6,F1) 'now, n_s = ',n_s
   write(6,F1) 'now, cnt = ',cnt

   deallocate(wgt)
   deallocate(rdist)
   deallocate(iind)
   deallocate(jind)

   !-----------------------------------------------
   ! resize the mapping, now that its size is known
   !-----------------------------------------------
   allocate(s(n_s))
   allocate(row(n_s))
   allocate(col(n_s))
   s = map%s(1:n_s)
   row = map%row(1:n_s)
   col = map%col(1:n_s)
   deallocate(map%s)
   deallocate(map%row)
   deallocate(map%col)
   map%n_s = n_s
   allocate(map%s(n_s))
   allocate(map%row(n_s))
   allocate(map%col(n_s))
   map%s = s
   map%row = row
   map%col = col
   deallocate(s)
   deallocate(row)
   deallocate(col)
   write(6,F1) 'leaving smooth'

END SUBROUTINE smooth

!===============================================================================

integer FUNCTION iadd(i,di,ni)

   implicit none

   !--- arguments ---
   integer :: i, di, ni

   if (di < 0) then
     iadd = mod(i+di-1+ni,ni)+1
   else
!    why was this coded like this??
!    iadd = mod(i,ni)+di
     iadd = mod(i+di,ni)
   endif

END FUNCTION iadd

!===============================================================================

integer FUNCTION jadd(j,dj,nj)

   implicit none

   !--- arguments ---
   integer :: j, dj, nj

   if (dj < 0) then
     jadd = max(j+dj,1)
   else
     jadd = min(j+dj,nj)
   endif

END FUNCTION jadd

!===============================================================================

real FUNCTION distance(x0,y0,x1,y1)

   implicit none

   !--- arguments ---
   real, intent(in) :: x0,y0,x1,y1

   !--- local ---
   real :: dph,dth,myx0,myx1

!-------------------------------------------------------------------------------
! PURPOSE:
!  o Return the distance in meters between two points on the
!    Earth with coordinates (x0,y0) and (x1,y1) in degrees.
!  o NOTE: does not take into account curvature of the grid...
!-------------------------------------------------------------------------------

   myx0 = x0
   myx1 = x1
   if (x1-x0>180.0) myx0 = x0 + 360.0
   if (x0-x1>180.0) myx1 = x1 + 360.0
   dph  = (myx1-myx0)*cos(DtoR*(y0+y1)/2.0)
   dth  = (y1-y0)
   distance = sqrt(dth**2 + dph**2)*DtoR*rEarth

END FUNCTION distance

!===============================================================================

real FUNCTION falloff(x1,y1,x2,y2,s)

   implicit none

   !--- arguments ---
   real, intent(in) :: x1,y1,x2,y2,s
   real :: dr

!-------------------------------------------------------------------------------
! PURPOSE:
!  o Compute exponential fall-off factor given two lon/lat
!    oordinates (x1,y1) and (x2,y2) in degrees.
!  o NOTE: does not take into account curvature of the grid...
!  o Function is g(d) = exp(-d/s)
!                       for distance d, efold scale length s (m)
!-------------------------------------------------------------------------------

    dr = distance(x1,y1,x2,y2)
    falloff = exp(-dr/s)

end FUNCTION falloff

!===============================================================================

SUBROUTINE test_smooth(map,field,value,source)

   implicit none

   !--- arguments ---
   type(sMatrix), intent(in) :: map  	! smoothing map     
   real, intent(inout) :: field(:,:)	! output flux field
   real, intent(in) :: value		! source flux value
   integer, intent(in) :: source(:,:)	! source points

   !--- local ---
   integer :: i,n,isrc,jsrc,idst,jdst
   integer :: nsrc
   integer, allocatable :: mask(:,:)

   nsrc = size(source,2)
   field = 0.0
   allocate(mask(map%ni_b,map%nj_b))
   mask = reshape(map%mask_b,(/map%ni_b,map%nj_b/))
   do n=1,map%n_s
       isrc = mod(map%col(n)-1,map%ni_a)+1
       jsrc = int((map%col(n)-1)/map%ni_a)+1
       idst = mod(map%row(n)-1,map%ni_b)+1
       jdst = int((map%row(n)-1)/map%ni_b)+1
       do i=1,nsrc
         if (isrc == source(1,i) .and. jsrc == source(2,i)) then
            field(idst,jdst) = field(idst,jdst)+map%s(n)*value
         endif
       enddo
   end do

   where(mask == 0) field = 1.e30

END SUBROUTINE test_smooth

!===============================================================================

SUBROUTINE smooth_field_write(map,field,filename)

   implicit none

   !--- arguments ---
   type(sMatrix), intent(in) :: map       ! smoothing map
   real, intent(in) :: field(:,:)         ! output flux field
   character(*) , intent(in) :: filename  ! name of data file

   !--- local ---
   integer :: i,m,n,isrc,jsrc,idst,jdst
   integer :: nsrc
   real    :: spv			  ! field missing value
   character(len= 8)     :: cdate   ! wall clock date
   character(len=10)     :: ctime   ! wall clock time
   character(len=240)     :: str     ! variable length char string
   character(len=240)     :: attstr  ! netCDF attribute name string
   integer                :: rcode   ! netCDF routine return code
   integer                :: fid     ! netCDF file      ID
   integer                :: vid     ! netCDF variable  ID
   integer                :: did     ! netCDF dimension ID
   integer                :: vdid(2) ! netCDF dimension ID

   !--- formats ---
   character(len=*),parameter :: F00 = "('(map_write) ',3a)"

!-------------------------------------------------------------------------------
! PURPOSE:
! o writes a field computed from smoothing "map" for viewing
! o assumes field grid is equivalent to b grid of "map"
!-------------------------------------------------------------------------------
   
   spv = maxval(field)

   !-----------------------------------------------------------------
   ! create a new nc file
   !-----------------------------------------------------------------
   rcode = nf_create(trim(filename),NF_CLOBBER,fid)
   if (rcode.ne.NF_NOERR) write(*,F00) nf_strerror(rcode)

   !-----------------------------------------------------------------
   ! global attributes
   !-----------------------------------------------------------------
    str  = map%title
   rcode = nf_put_att_text(fid,NF_GLOBAL,'title'      ,len_trim(str),str)
    str  = map%normal
   rcode = nf_put_att_text(fid,NF_GLOBAL,'normalization',len_trim(str),str)
    str  = map%method
   rcode = nf_put_att_text(fid,NF_GLOBAL,'map_method' ,len_trim(str),str)
    str  = map%history
   call date_and_time(cdate,ctime) ! f90 intrinsic
    str = 'File created: '//cdate(1:4)//'-'//cdate(5:6)//'-'//cdate(7:8) &
   &                 //' '//ctime(1:2)//':'//ctime(3:4)//':'//ctime(5:6)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'history'    ,len_trim(str),str)
    str  = map%convention
   rcode = nf_put_att_text(fid,NF_GLOBAL,'conventions',len_trim(str),str)
    str  = map%domain_a
   rcode = nf_put_att_text(fid,NF_GLOBAL,'domain_a'   ,len_trim(str),str)
    str  = map%domain_b
   rcode = nf_put_att_text(fid,NF_GLOBAL,'domain_b'   ,len_trim(str),str)

   !-----------------------------------------------------------------
   ! dimension data
   !-----------------------------------------------------------------
   rcode = nf_def_dim(fid, 'ni_b', map%ni_b, did) ! # of points wrt i
   rcode = nf_def_dim(fid, 'nj_b', map%nj_b, did) ! # of points wrt j

   !-----------------------------------------------------------------
   ! define data 
   !-----------------------------------------------------------------

   rcode = nf_inq_dimid(fid,'ni_b',vdid(1))
   rcode = nf_inq_dimid(fid,'nj_b',vdid(2))
   rcode = nf_def_var  (fid,'field',NF_DOUBLE,2,vdid,vid)
   str   = 'output of call to test_smooth'
   rcode = nf_put_att_text(fid,vid,"description",len_trim(str),str)
   rcode = nf_put_att_double(fid,vid,"missing_value",NF_FLOAT,1,spv)

   !-----------------------------------------------------------------
   ! put data
   !-----------------------------------------------------------------
   rcode = nf_enddef(fid)

   rcode = nf_inq_varid     (fid,  'field',vid)
   rcode = nf_put_var_double(fid, vid, field) 

   rcode = nf_close(fid)

   if (rcode.ne.NF_NOERR) write(*,F00) nf_strerror(rcode)

END SUBROUTINE smooth_field_write

!===============================================================================

SUBROUTINE dummyflux(imt,jmt,mask,xt,yt,at,runfile,efold)

!-------------------------------------------------------------------------------
!    Based on gconst of /home/dataproc/yeager/RUNOFF/fortran/garray.f
!    which was invoked as a shared object by the NCL routine
!    /home/dataproc/yeager/RUNOFF/construct_runoff_netcdf_dummy.ncl
!
! PURPOSE:
!  o Construct runoff mapping based on a 'runfile' data file which
!    partitions known river runoffs into a 19-basin scheme by Large. 
!  o runfile specifies rivers/coastal discharges (kg/s) for each basin along
!    with lon/lat boxes within which this discharge is to be distributed.
!    The distribution array is g(:,:,:) with units (kg/s/m^2).
!    This routine distributes 'coastal' runoff as:
!
!	 g(i,j,thisbasin) = [total coastal discharge (kg/s)]/[total
!				distribution area (m^2)]		
!
!    This routine distributes 'river' runoff as:
!
!        g(i,j,thisriver) = [exponential falloff factor]*[total river 
!				discharge (kg/s)]/[total exponential-weighted
!				distribution area (m^2)]     
!
!    Areas (from array at) are based on ocean grid TAREA.
!  
!-------------------------------------------------------------------------------

   !--- local ---
   integer :: imt,jmt,jm,jp,im,ip,nr,n,i,j
   real    :: rlat,rlon,xf
   integer, dimension(imt,jmt) :: mask	! mask of grid (0 for land)
   real, dimension(imt,jmt) :: xt,yt	! lon,lat of grid
   real, dimension(imt,jmt) :: at	! area of grid (m^2)
   real, dimension(imt,jmt) :: A	! masked out area array
   real, dimension(imt,jmt) :: fi	! exponential factor
   real, allocatable :: total(:)	! total runoff by basin (kg/s)
   character*47 :: runfile
   character*27 :: rname
   integer, parameter :: nrmax=70	! max number of rivers
   integer :: nb,nbo,ibasin,nrivers,msea
   real :: efold			! efold scale (m) for falloff
   real :: rarea			! sum of destination cell areas
   real :: firarea			! exponential-weighted sum of 
					! destination cell areas
   real :: sumb,sumd,gmax
   real, dimension(2,nrmax) :: blon,blat
   real, dimension(nrmax) :: clon,clat,nref,discharge
   character*40, allocatable :: bname(:)	

   ! Check that received grid lon/lat arrays are reasonable
   if (minval(xt).lt.0.0 .or. maxval(xt).gt.360.0) then
      write(6,*) 'xt out of range'
      stop
   endif
   if (minval(yt).lt.-90.0 .or. maxval(yt).gt.90.0) then
      write(6,*) 'yt out of range'
      stop
   endif

   open(13,file=runfile,status='old',form='formatted')
   read(13,*)
   read(13,*) nbo
   nb = nbo+1		! nbo seperate basins, plus global
   allocate(garr(imt,jmt,nb))
   allocate(total(nb))
   allocate(bname(nb))
   total = 0.0
   garr = 0.0

   bname(1) = '    Global'

   ! Process each of the basins
   do n=2,nb
     read(13,99) ibasin,nrivers,msea,clon(1),clat(1),bname(n)
     sumd = 0.0
     do nr=1,nrivers
       read (13,100) nref(nr),rname,discharge(nr),clon(nr),clat(nr), &
     &        blon(1,nr),blon(2,nr),blat(1,nr),blat(2,nr)
       sumd = sumd + discharge(nr)
     enddo
     total(n) = sumd	! includes coastal runoff

     do nr=1,nrivers
       xf = discharge(nr) / sumd	
       rarea = 0.0
       firarea = 0.0
       do j=1,jmt
         do i=1,imt
!             m=(j-1) * (imt-2) + i - 1
           rlon = xt(i,j)
           rlat = yt(i,j)
           if (blon(1,nr).lt.0.0.and.xt(i,j).gt.180.) rlon = xt(i,j)-360.0

           ! If the grid point lies in the specified lon/lat box
	   ! where the discarge is to be distributed...
	   if ((blat(1,nr).le.rlat).and.(rlat.le.blat(2,nr)).and. &
     &	       (mask(i,j).ne.0).and.(blon(1,nr).le.rlon).and. &
     &         (rlon.le.blon(2,nr))) then

	      if (nref(nr).eq.0) then	! for Coastal runoff
					! check that destination cell
					! is adjacent to land
                jm = jadd(j,-1,jmt)
                jp = jadd(j,1,jmt)
                im = iadd(i,-1,imt)
                ip = iadd(i,1,imt)
!	exception for (OLD) grids with wraparound points
	if ((imt.eq.102.or.imt.eq.152).and.i.eq.1) im = imt-1
	if ((imt.eq.102.or.imt.eq.152).and.i.eq.imt) ip = 2

		if ((mask(im,j).eq.0).or.(mask(im,jp).eq.0) &
     & .or.(mask(i,jp).eq.0).or.(mask(ip,jp).eq.0).or. &
     & (mask(ip,j).eq.0).or.(mask(ip,jm).eq.0).or. &
     & (mask(i,jm).eq.0).or.(mask(im,jm).eq.0)) then 
		  A(i,j) = at(i,j)
		  rarea = rarea + A(i,j)
		else
		  A(i,j) = 0.0
                endif
	      else			! for River runoff
                 A(i,j)  = at(i,j)
                 rarea = rarea + A(i,j)     
		 fi(i,j) = falloff(rlon,rlat,clon(nr),clat(nr),efold)
                 firarea = firarea + fi(i,j)*A(i,j)     
	      endif
           else
              A(i,j)  = 0.0
              fi(i,j)  = 0.0
           endif
         enddo
       enddo

       do j=1,jmt
         do i=1,imt
           !  for unitless 
           !  garr(i,j,n) = garr(i,j,n) + xf * A(i,j) / rarea
           !  for [kg/m^2/s]
           if (A(i,j) .gt. 0.) then
	     if (nref(nr).eq.0) then
	       garr(i,j,n) = garr(i,j,n) + discharge(nr)/rarea
	     else
	       garr(i,j,n) = garr(i,j,n) + fi(i,j)*discharge(nr) / firarea
	     endif
	   endif
         enddo
       enddo

     enddo     ! do nr=1,nrivers
   enddo       ! do n=2,nb

! Compute global total runoff
   do n=2,nb
     total(1) = total(1) + total(n)
   enddo
   do j=1,jmt
     do i=1,imt
       do n=2,nb
          garr(i,j,1) = garr(i,j,1) + garr(i,j,n)
       enddo
     enddo
   enddo

! Test runoff conservation and write summary table to runoff.doc
   write(6,*) 'Results of Data Runoff Distribution:'
   write(6,101) 'Basin','Global g*area (kg/s)','Data (kg/s)','Difference'
   do n=1,nb
     gmax = 0.0
     sumb = 0.0
     do j=1,jmt
       do i=1,imt
         sumb = sumb + garr(i,j,n)*at(i,j)
	 if (garr(i,j,n).gt.gmax) gmax = garr(i,j,n)
       enddo
     enddo
     write(6,102) n,bname(n),sumb,total(n),(sumb - total(n))
   enddo

99     format(3i4,2f8.1,a30)
100    format(i4,2x,a27,f12.1,6f7.1)
101    format(a10,11x,a25,4x,a11,4x,a17)
102    format(i2,2x,a26,2x,f14.3,3x,f14.3,3x,f14.3)

END SUBROUTINE dummyflux

!===============================================================================

SUBROUTINE dummyweights(map,at)

   implicit none

   !--- arguments ---
   type(sMatrix), intent(inout) :: map  ! scrip map template to be filled
   real		:: at(:,:)		! ocn TAREA in m^2

   !--- local ---
   integer :: ni,nj,imt,jmt,i,j,m,n_s,nbasin
   real    :: sumb
   real, allocatable  :: col(:,:),row(:,:),s(:,:),garr2d(:,:)
   real, allocatable  :: col1d(:),row1d(:),s1d(:),cplratio(:),at1d(:)
   logical, allocatable  :: smask(:,:)
   real, allocatable  :: total(:)

!-------------------------------------------------------------------------------
! PURPOSE:
! o Based on M. Hecht's /fs/cgd/home0/hecht/csm/runoff/transfer_matrix.ncl
! o Compute correct weights to map 19-element 'normalized' runoff flux vector 
!   (kg/s/m^2) onto the POP ocean grid.  
!
!	R(j)  = net runoff (kg/s) into basin j
!	dA(j) = fictitious area (m^2) associated with basin j (1/19 of globe)
!	        from scrip
!       F(j)  = "flux" sent to cpl by dlnd = R(j)/dA(j) (kg/s/m^2)
!		(This flux is stored in the data_aroff netcdf file)
!       garr(i)  = flux which should go into ocean cell i, as determined by
!	         routine dummyflux (sum over i is known to preserve R when
!		 multiplied by ocean areas!)
!       dA_o(i) = area (m^2) associated with ocean cell i, from POP
!       dA_s(i) = area (m^2) associated with ocean cell i, from scrip
!
!   Then in order to preserve R(i), the mapping weights should be
!       s(i,j) = dA(j)*[garr(i)/R(j)]*[dA_o(i)/dA_s(i)]
!
!   The "flux" mapping, then will be:
!	Fo(i) = sum_over_j{s(i,j)*F(j)} 
!	      = sum_over_j{dA(j)   garr(i)    dA_o(i)   R(j)  }
!		          {     * --------- * ------- * ----  }
!			  {        R(j)	      dA_s(i)   dA(j) }
!	      = garr(i) * dA_o(i)/dA_s(i)
!
!   Before sending to the ocean, the couple will multiply Fo(i) by
!   an area correction factor, so that
!	F*o(i) = runoff flux received by ocean model (kg/s/m^2)
!	       = Fo(i) * dA_s(i)/dA_o(i)
!	       = garr(i), by definition the correct flux into ocean cell i
!
!   Therefore, sum_over_i{F*o(i)*dAo(i)} = sum_over_j{R(j)}
!	         
!-------------------------------------------------------------------------------

   imt = map%ni_b
   jmt = map%nj_b
   ni = map%ni_a
   nj = map%nj_a
   
   allocate(col(map%n_b,map%n_a))
   allocate(row(map%n_b,map%n_a))
   allocate(s(map%n_b,map%n_a))
   allocate(total(map%n_a))
   allocate(smask(map%n_b,map%n_a))
   nbasin = size(garr,3) 
   allocate(garr2d(map%n_b,nbasin-1))
   allocate(cplratio(map%n_b))
   allocate(at1d(map%n_b))

   ! This is the areafact ratio that cpl will multiply
   ! runoff flux with before sending to ocean
   cplratio = (map%area_b*(rEarth**2))/reshape(at,(/map%n_b/))
   
   garr2d = reshape(garr(:,:,2:nbasin),(/map%n_b,nbasin-1/))
   at1d = reshape(at,(/map%n_b/))

   !-----------------------------------------------------------------
   ! compute the 19-basin total runoffs (kg/s)
   !-----------------------------------------------------------------
   do j=1,map%n_a
     sumb = 0.0
     do i=1,map%n_b
       sumb = sumb + garr2d(i,j)*at1d(i)
     enddo
     total(j) = sumb
   enddo

   ! Using garr, fill in the mapping matrix
   do i=1,map%n_b
     do j=1,map%n_a
       col(i,j) = j
       row(i,j) = i
       s(i,j) = map%area_a(j)*(rEarth**2)*garr2d(i,j)/(total(j)*cplratio(i))
    !  s(i,j) = map%area_a(j)*(rEarth**2)*garr2d(i,j)/(total(j))
     enddo
   enddo

   !-----------------------------------------------
   ! resize the mapping, now that its size is known
   !-----------------------------------------------
   smask = .false.
   where(s.gt.0.0) smask=.true.
   n_s = count(smask)
   allocate(s1d(n_s))
   allocate(row1d(n_s))
   allocate(col1d(n_s))
   s1d = pack(s,smask)
   col1d = pack(col,smask)
   row1d = pack(row,smask)

   deallocate(map%s)
   deallocate(map%row)
   deallocate(map%col)
   map%n_s = n_s
   allocate(map%s(n_s))
   allocate(map%row(n_s))
   allocate(map%col(n_s))
   map%s = s1d
   map%row = row1d
   map%col = col1d

END SUBROUTINE dummyweights

!===============================================================================

SUBROUTINE dummy_aroff_write(map,srcfile,datafile,at)

   implicit none

   !--- arguments ---
   type(sMatrix), intent(in) :: map       ! scrip map template
   character(*) , intent(in) :: srcfile   ! name of ascii river runoff file
   character(*) , intent(in) :: datafile  ! name of runoff data file
   real , intent(in) :: at(:,:)		  ! ocean TAREA (m^2)

   !--- local ---
   integer :: ni,nj,imt,jmt,i,j,m,n
   real    :: sumb
   real, allocatable     :: work(:,:),work2(:,:,:),basinroff(:,:)
   integer, allocatable  :: iwork(:,:)
   character(len= 8)     :: cdate   ! wall clock date
   character(len=10)     :: ctime   ! wall clock time
   character(len=240)     :: str     ! variable length char string
   character(len=240)     :: attstr  ! netCDF attribute name string
   integer                :: rcode   ! netCDF routine return code
   integer                :: fid     ! netCDF file      ID
   integer                :: vid     ! netCDF variable  ID
   integer                :: did     ! netCDF dimension ID
   integer                :: vdid(3) ! netCDF dimension ID

   !--- formats ---
   character(len=*),parameter :: F00 = "('(dummy_aroff_write) ',3a)"

!-------------------------------------------------------------------------------
! PURPOSE:
! o Once garr has been defined after a call to dummyflux, this routine
!   will output a data.runoff.nc file for 19-basin data runoff.  This
!   file is needed by dlnd6, and both "domain.runoff.nc" and "data.runoff.nc"
!   get softlinked to this file in the dlnd buildnml script.  The dlnd
!   namelist parameter data_aroff will get set to this file.  
!
! o The contents of the netcdf are fictitious 19-basin grid information (derived
!   from the r19.nc file through the sMatrix "map") and runoff values for each 
!   basin in kg/s/m^2.  These values are computed as the total runoff for each 
!   basin as specified in srcfile (kg/s) normalized by a fictitious area 
!   (1/19 of the globe) in m^2.
!-------------------------------------------------------------------------------
   
   imt = map%ni_b
   jmt = map%nj_b
   ni = map%ni_a
   nj = map%nj_a
   allocate(iwork(ni,nj))
   allocate(work(ni,nj))
   allocate(work2(ni,nj,4))
   allocate(basinroff(ni,nj))

   !-----------------------------------------------------------------
   ! compute the normalized 19-basin totals (kg/s/m^2)
   ! (This is total of each of 19 basins (kg/s) divided by bogus
   !  'basin area' = (area of Earth)/19 ).
   !-----------------------------------------------------------------
   do n=1,ni
     sumb = 0.0
     do j=1,jmt
       do i=1,imt
         sumb = sumb + garr(i,j,n+1)*at(i,j)
       enddo
     enddo
     basinroff(n,1) = sumb/(map%area_a(n)*rEarth*rEarth)
   enddo

   !-----------------------------------------------------------------
   ! create the 19-basin data_aroff netcdf used by dlnd6
   !-----------------------------------------------------------------
   rcode = nf_create(trim(datafile),NF_CLOBBER,fid)
   if (rcode.ne.NF_NOERR) write(*,F00) nf_strerror(rcode)

   !-----------------------------------------------------------------
   ! global attributes
   !-----------------------------------------------------------------
    str  = 'runoff data, fictitious 19 basin domain'
   rcode = nf_put_att_text(fid,NF_GLOBAL,'title'      ,len_trim(str),str)
   call date_and_time(cdate,ctime) ! f90 intrinsic
    str = 'File created: '//cdate(1:4)//'-'//cdate(5:6)//'-'//cdate(7:8) &
   &                 //' '//ctime(1:2)//':'//ctime(3:4)//':'//ctime(5:6)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'history'    ,len_trim(str),str)
    str  = 'created using dummy runoff tools in /cgd/oce/yeager/POP_tools/'// &
   &	'runoff/runoff_smooth_mod.F90'
   rcode = nf_put_att_text(fid,NF_GLOBAL,'creation',len_trim(str),str)
    str  = 'source data file was '//trim(srcfile)
   rcode = nf_put_att_text(fid,NF_GLOBAL,'data'   ,len_trim(str),str)

   !-----------------------------------------------------------------
   ! dimensions
   !-----------------------------------------------------------------
   rcode = nf_def_dim(fid, 'ni', ni, did) ! # of points wrt i
   rcode = nf_def_dim(fid, 'nj', nj, did)  ! # of points wrt j
   rcode = nf_def_dim(fid, 'nv', 4, did)  ! # of points wrt j

   !-----------------------------------------------------------------
   ! variables
   !-----------------------------------------------------------------
   rcode = nf_inq_dimid(fid,'nv',vdid(1))
   rcode = nf_inq_dimid(fid,'ni',vdid(2))
   rcode = nf_inq_dimid(fid,'nj',vdid(3))
   rcode = nf_def_var(fid,'xc',NF_DOUBLE,2,vdid(2:3),vid)
   str   = 'degrees'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'longitude of runoff grid cell center'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_def_var(fid,'yc',NF_DOUBLE,2,vdid(2:3),vid)
   str   = 'degrees'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'latitude of runoff grid cell center'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_def_var(fid,'xv',NF_DOUBLE,3,vdid,vid)
   str   = 'degrees'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'longitudes of runoff grid cell vertices'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_def_var(fid,'yv',NF_DOUBLE,3,vdid,vid)
   str   = 'degrees'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'latitudes of runoff grid cell vertices'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_def_var(fid,'mask',NF_SHORT,2,vdid(2:3),vid)
   str   = 'unitless'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'runoff grid domain mask'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_def_var(fid,'area',NF_DOUBLE,2,vdid(2:3),vid)
   str   = 'square radians'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'area of runoff grid cell'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_def_var(fid,'runoff',NF_DOUBLE,2,vdid(2:3),vid)
   str   = 'kg/s/m^2'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'Basin total runoff normalized by 1/19 of Globe'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)
   rcode = nf_put_att_double(fid,vid,"_FillValue",NF_DOUBLE,1,-9999.)
   rcode = nf_def_var(fid,'rEarth',NF_DOUBLE,0,vdid,vid)
   str   = 'meters'
   rcode = nf_put_att_text(fid,vid,'units',len_trim(str),str)
   str   = 'Radius of the Earth'
   rcode = nf_put_att_text(fid,vid,'long_name',len_trim(str),str)

   !-----------------------------------------------------------------
   ! finish writing it
   !-----------------------------------------------------------------
   rcode = nf_enddef(fid)
   rcode = nf_inq_varid(fid,'xc',vid)
   work = reshape(map%xc_a,(/ni,nj/))
   rcode = nf_put_var_double(fid, vid, work) 
   rcode = nf_inq_varid(fid,'yc',vid)
   work = reshape(map%yc_a,(/ni,nj/))
   rcode = nf_put_var_double(fid, vid, work) 
   rcode = nf_inq_varid(fid,'xv',vid)
   work2 = reshape(map%xv_a,(/ni,nj,4/))
   rcode = nf_put_var_double(fid, vid, work2) 
   rcode = nf_inq_varid(fid,'yv',vid)
   work2 = reshape(map%yv_a,(/ni,nj,4/))
   rcode = nf_put_var_double(fid, vid, work2) 
   rcode = nf_inq_varid(fid,'mask',vid)
   iwork = reshape(map%mask_a,(/ni,nj/))
   rcode = nf_put_var_int(fid, vid, iwork) 
   rcode = nf_inq_varid(fid,'area',vid)
   work = reshape(map%area_a,(/ni,nj/))
   rcode = nf_put_var_double(fid, vid, work)
   rcode = nf_inq_varid(fid,'rEarth',vid)
   rcode = nf_put_var_double(fid, vid, rEarth)
   rcode = nf_inq_varid(fid,'runoff',vid)
   rcode = nf_put_var_double(fid, vid, basinroff)
   rcode = nf_close(fid)
   if (rcode.ne.NF_NOERR) write(*,F00) nf_strerror(rcode)

END SUBROUTINE dummy_aroff_write

!===============================================================================

SUBROUTINE POP_TAREA_compute(filename,nx,ny,area)

   implicit none

   !--- arguments ---
   character(*) , intent(in)    :: filename  ! name of ocn grid binary
   integer	:: nx,ny
   real , allocatable, intent(out)   :: area(:,:)

   !--- local ---
   real , allocatable   :: work(:,:),htn(:,:),hte(:,:),dxt(:,:),dyt(:,:)

   !--- formats ---
   character(len=*),parameter :: F00 = "('(POP_TAREA_compute) ',3a)"

!-------------------------------------------------------------------------------
! PURPOSE:
! o Compute POP TAREA array from grid.ieeer8 binary, as it is done
!   in the POP code.
!-------------------------------------------------------------------------------

   allocate(work(nx,ny))
   allocate(htn(nx,ny))
   allocate(hte(nx,ny))
   allocate(dxt(nx,ny))
   allocate(dyt(nx,ny))
   allocate(area(nx,ny))

   open(10,file=filename,form='unformatted',access='direct', &
  &	recl=nx*ny*8)
   read(10,rec=1) work
   read(10,rec=2) work
   read(10,rec=3) htn
   read(10,rec=4) hte
   close(10)

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!  Below code fragment is from POP grid.F
!  slightly modified
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
      !***
      !*** construct T-grid cell widths
      !***
      work = cshift(htn,-1,2)
      dxt = 0.5*(htn + work)
      dxt(:,1) = dxt(:,2)	! reasonable kluge

      work = cshift(hte,-1,1)
      dyt = 0.5*(hte + work)

      where (dxt == 0.0) dxt=1.0
      where (dyt == 0.0) dyt=1.0

      area = dxt*dyt
 
END SUBROUTINE POP_TAREA_compute

!===============================================================================

SUBROUTINE POP_2D_read(filename,varname,data)

   !--- modules ---

   implicit none

   !--- includes ---
#include <netcdf.inc>

   !--- arguments ---
   character(*) , intent(in)    :: filename  ! name of data file
   character(*) , intent(in)    :: varname  ! name of data file
   real*4 , intent(inout)         :: data(:,:)

   !--- local ---
   integer         :: nxin,nyin,nx,ny,ndims,itype     ! generic indicies
   integer         :: fid,vid,did,rcode
   integer, dimension(2)  :: dimid

   !--- formats ---
   character(len=*),parameter :: F00 = "('(POP_2D_read) ',3a)"

!-------------------------------------------------------------------------------
! PURPOSE:
! o reads map matrix information from netCDF data file
!
! NOTE:
!-------------------------------------------------------------------------------

   nxin = size(data,1)
   nyin = size(data,2)

   !-----------------------------------------------
   ! read in the variable
   !-----------------------------------------------
   rcode = nf_open(filename,NF_NOWRITE,fid)
   rcode = nf_inq_varid(fid,varname,vid)
   if (rcode.ne.NF_NOERR) write(*,F00) nf_strerror(rcode)
   rcode = nf_inq_varndims(fid,vid,ndims)
   if (ndims.ne.2) then 
	write(*,F00) 'variable did not have dim 2'
	stop
   endif
   rcode = nf_inq_vardimid(fid,vid,dimid)
   rcode = nf_inq_dimlen(fid, dimid(1)   , nx  )
   rcode = nf_inq_dimlen(fid, dimid(2)   , ny  )
   if (nx.ne.nxin.or.ny.ne.nyin) then 
	write(*,F00) 'variable does not have correct size',nx,ny
	stop
   endif
   rcode = nf_inq_vartype(fid,vid,itype)

   select case (itype)
   case (NF_FLOAT)
      rcode = nf_get_var_real(fid,vid,data)
   case default
      write(*,F00) 'can only handle float'
      stop
   end select

   rcode = nf_close(fid)

END SUBROUTINE POP_2D_read

!=================================================================

recursive SUBROUTINE recur_setDist(i0,j0,ni,nj,x,y,value,mask,rdist,rmax)

!-----------------------------------------------------------------------
!  Recursively sets a distance from a seed point (i0,j0)
!-----------------------------------------------------------------------

   integer :: i0, j0, ni, nj
   integer :: ie,iw,jn,js,k,j
   integer, dimension(ni,nj) :: mask
   real, dimension(ni,nj) :: rdist
   real :: rmax,d,value
   real, dimension(ni*nj) :: x,y

   if (mask(i0,j0) /= 0 .and. value < rmax .and. value < rdist(i0,j0)) then
      j = (j0-1)*ni+i0
      rdist(i0,j0)=value
      mask(i0,j0)=1

      iw = iadd(i0,-1,ni)
      k = (j0-1)*ni+iw
      d = distance(x(k),y(k),x(j),y(j))+rdist(i0,j0)
      call recur_setDist(iw,j0,ni,nj,x,y,d,mask,rdist,rmax)

      ie = iadd(i0,1,ni)
      k = (j0-1)*ni+ie
      d = distance(x(k),y(k),x(j),y(j))+rdist(i0,j0)
      call recur_setDist(ie,j0,ni,nj,x,y,d,mask,rdist,rmax)

      jn = jadd(j0,1,nj)
      k = (jn-1)*ni+i0
      d = distance(x(k),y(k),x(j),y(j))+rdist(i0,j0)
      call recur_setDist(i0,jn,ni,nj,x,y,d,mask,rdist,rmax)

      js = jadd(j0,-1,nj)
      k = (js-1)*ni+i0
      d = distance(x(k),y(k),x(j),y(j))+rdist(i0,j0)
      call recur_setDist(i0,js,ni,nj,x,y,d,mask,rdist,rmax)

   endif

END SUBROUTINE recur_setDist

!=================================================================


!===============================================================================
END MODULE smooth_mod
!===============================================================================
