!===============================================================================
! SVN $Id: main.F90 2892 2007-01-12 23:31:02Z kauff $
! SVN $URL: https://svn-ccsm-models.cgd.ucar.edu/tools/mapping/gen_runoffmap/trunk/src/main.F90 $
!===============================================================================

PROGRAM main

   use shr_timer_mod
   use kind_mod

   use map_mod
   use mapsort_mod
   use fixroff_mod
   use smooth_mod

   implicit none

   !--- mapping matricies ---
   type(sMatrix),target :: map_orig    ! orig matrix -- produced by scrip, may have runoff over land
   type(sMatrix),target :: map_corr    ! orig matrix, corrected -- no runoff over land
   type(sMatrix),target :: map_new     ! orig matrix, corrected, smoothed -- unsorted & sorted
   type(sMatrix),target :: map_smooth  ! smoothing matrix

   !--- for correction (runoff relocations) and smoothing ---
   integer(IN)       :: n               !
   real(R8)          :: eFold           ! smoothing e-fold curve
   real(R8)          :: rMax            ! max smoothing radius

   integer(IN)       :: t01,t02,t03,t04,t05,t06 ! timer numbers
   integer(IN)       :: rCode           ! return code
   character( 8)     :: dstr            ! wall clock date
   character(10)     :: tstr            ! wall clock time

   logical,parameter :: step1  = .true.
   logical,parameter :: step2  = .true.
   logical,parameter :: step3a = .true.
   logical,parameter :: step3b = .true.

   !--- namelis vars ---
   character(180) :: file_orig     ! file name: orig matrix
   character(180) :: file_corr     ! file name: orig matrix, corrected
   character(180) :: file_unsorted ! file name: orig matrix, corrected, smoothed -- unsorted
   character(180) :: file_new      ! file name: orig matrix, corrected, smoothed, sorted -- done
   character(180) :: file_smooth   ! file name: smoothing matrix
   character(180) :: file_sources  ! file name: mask that restricts set of runoff source points
   character(180) :: title         ! netCDF title attribute

   namelist / input_nml / &
      file_orig     &
   ,  file_corr     &
   ,  file_unsorted &
   ,  file_new      &
   ,  file_smooth   &
   ,  file_sources  &
   ,  title         &
   ,  eFold         &
   ,  rMax          

   !--- formats ---
   character(*),parameter :: F00 = "('(main) ',6a)"
   character(*),parameter :: F01 = "('(main) ',a,i7)"
   character(*),parameter :: F02 = "('(main) ',a,es13.6)"
   character(*),parameter :: F10 = "('(main) ',73('-')/,'(main) ',a/,'(main) ',73('-'))"
   character(*),parameter :: F12 = "('(main) date & time:',1x,a4,2('-',a2),2x,a2,2(':',a2))"

!-------------------------------------------------------------------------------
! PURPOSE:
!   Take a scrip conservative map matrix, runoff->ocean (unmasked ocean)
!   and produce a runoff->ocean map suitable for use in CCSM
! 
! Step 1: Correct original scrip mapping (and sort it)
!         if runoff is mapped to a land cell, move it to an ocean cell
!         - 2a: relocate (correct) destination cells in masked areas
!         - 2b: sort the corrected map
! Step 2: Create the smoothing map (and sort it)
!         smoothing map determines how runoff gets spread to more ocean cells
!         - 2a: create smoothing map
!         - 2b: sort   smoothing map
! Step 3: Multiply corrected original with smoothing (and sort new map)
!         - 3a: a matrix multiply applies the smoothing to the corrected runoff map
!         - 3b: sort the new matrix
!-------------------------------------------------------------------------------

   write(6,F10) "correct/smooth/sort runoff -> ocean map"
   write(6,F10) "SVN $URL: https://svn-ccsm-models.cgd.ucar.edu/tools/mapping/gen_runoffmap/trunk/src/main.F90 $"

   call shr_timer_init()

   !----------------------------------------------------------------------------
   write(6,F10) "Step 0:  read input namelist data"
   !----------------------------------------------------------------------------

   file_orig     = 'unset'
   file_corr     = 'unset'
   file_smooth   = 'unset'
   file_unsorted = 'unset'
   file_sources  = 'unset'
   file_new      = 'unset'
   title         = 'unset'
   eFold         = 1000000.00000 ! smoothing eFold distance in meters
   rMax          =  500000.00000 ! max smoothing radius in meters

   open (10,file="runoff_map.nml",status="old",action="read")
   read (10,nml=input_nml,iostat=rCode)
   close(10)
   write(6,F00) "Namelist values..."
   write(6,F00) "   file_orig      = ",trim(file_orig     )
   write(6,F00) "   file_corr      = ",trim(file_corr     )
   write(6,F00) "   file_smooth    = ",trim(file_smooth   )
   write(6,F00) "   file_unsorted  = ",trim(file_unsorted )
   write(6,F00) "   file_sources   = ",trim(file_sources  )
   write(6,F00) "   file_new       = ",trim(file_new      )
   write(6,F00) "   title          = ",trim(title         )
   write(6,F02) "   eFold distance = ",eFold
   write(6,F02) "   rMax  distance = ",rMax
!  if (rCode > 0) then
!     write(6,F01) 'ERROR: reading input namelist, iostat=',rCode
!     stop
!  end if

if (step1) then  !~~~ 1a & 1b ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

   !----------------------------------------------------------------------------
   write(6,F10) "Step 1a:  Correct original scrip mapping"
   !----------------------------------------------------------------------------
   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call shr_timer_get  (t01,"Step 1a: Correct original scrip mapping")
   call shr_timer_start(t01)

   call map_read (map_orig , trim(file_orig))       ! read original scrip map
   call map_Sn1  (map_orig)
   call map_check(map_orig)

   call fixroff_modifyMap (map_orig, map_corr) ! mod map wrt relocated points

   call shr_timer_stop (t01)
   call shr_timer_print(t01)

   !----------------------------------------------------------------------------
   write(6,F10) "Step 1b:  Sort the corrected original scrip mapping"
   !----------------------------------------------------------------------------
   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call shr_timer_get  (t02,"Step 1b: Sort the corrected original scrip mapping")
   call shr_timer_start(t02)

   call mapsort_sort(map_corr)  ! sort map

   call shr_timer_stop (t02)
   call shr_timer_print(t02)

   call map_Sn1  (map_corr )
   call map_check(map_corr)

   call map_write(map_corr, trim(file_corr)) 

end if

if (step2) then  !~~~ 2a & 2b ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

   if (.not. step1) then
      call map_read (map_corr, trim(file_corr))  ! read corrected r2o map
      call map_Sn1  (map_corr)
      call map_check(map_corr)
   end if

   !----------------------------------------------------------------------------
   write(6,F10) " Step 2a:  Create the smoothing map"
   !----------------------------------------------------------------------------
   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call shr_timer_get  (t03,"Step 2a: Create the smoothing map")
   call shr_timer_start(t03)

   write(6,F02) "   eFold distance = ",eFold
   write(6,F02) "   rMax  distance = ",rMax

   call smooth_init(map_corr,map_smooth)
!  call restrictSources(map_smooth,trim(file_sources)) ! restrict cells subject to smoothing
   call smooth(map_smooth,eFold,rMax)

   call map_check(map_smooth)
   call map_write(map_smooth, trim(file_smooth)) 

   call shr_timer_stop (t03)
   call shr_timer_print(t03)

   !----------------------------------------------------------------------------
   write(6,F10) " Step 2b:  Sort the smoothing map"
   !----------------------------------------------------------------------------
   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call shr_timer_get  (t04,"Step 2b: Sort the smoothing map")
   call shr_timer_start(t04)

   call mapsort_sort(map_smooth)  ! sort map

   call shr_timer_stop (t04)
   call shr_timer_print(t04)

   call map_Sn1  (map_smooth)
   call map_check(map_smooth)
   call map_write(map_smooth, trim(file_smooth)) 

end if

if (step3a ) then  !~~~ 3a ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

   if (.not. step2) then
      call map_read (map_smooth, trim(file_smooth)) ! read smoothing map
      call map_Sn1  (map_smooth )
      call map_check(map_smooth)
      if (.not. step1) then
         call map_read (map_corr, trim(file_corr))  ! read corrected r2o map"
         call map_Sn1  (map_corr )
         call map_check(map_corr)
      end if
   end if

   !----------------------------------------------------------------------------
   write(6,F10) " Step 3a:  Apply smoothing map to runoff map "
   !----------------------------------------------------------------------------

   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call shr_timer_get  (t05,"Step 3a:  Apply smoothing map to runoff map")
   call shr_timer_start(t05)

   !--- create new map datatype to hold result of matrix-matrix multiply ---
   call map_dup(map_corr,map_new)
   map_new%title  = title
   deallocate( map_new%s  )
   deallocate( map_new%row)
   deallocate( map_new%col)
   n =   80*map_new%n_s     ! must guess a sufficient allocation size
   allocate(map_new%s  (n))
   allocate(map_new%row(n))
   allocate(map_new%col(n))
   map_new%n_s = 0          ! actual size will be set inside map_matmult

   call map_matMatMult(map_corr,map_new,map_smooth) ! mult(A,B,S): B=S*A

   call shr_timer_stop (t05)
   call shr_timer_print(t05)

   call map_Sn1  (map_new)
   call map_check(map_new)

   call map_write(map_new, trim(file_unsorted)) ! smoothed, but may need sorting
   
end if

if (step3b ) then  !~~~ 3b ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

   if (.not. step3a) then
      call map_read (map_new, trim(file_unsorted)) ! read smoothed/unsorted r2o map
      call map_Sn1  (map_new)
      call map_check(map_new)
   end if

   !----------------------------------------------------------------------------
   write(6,F10) " Step 3b:  Sort new smoothed runoff map "
   !----------------------------------------------------------------------------
   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call shr_timer_get  (t06,"Step 3b: Sort new smoothed runoff map")
   call shr_timer_start(t06)

   call mapsort_sort(map_new)  ! sort map

   call shr_timer_stop (t06)
   call shr_timer_print(t06)

   call map_check(map_new)

   !----------------------------------------------------------------------------
   write(6,F10) " Everything's done, save the final product"
   !----------------------------------------------------------------------------
   call date_and_time(dstr,tstr)
   write(6,F12) dstr(1:4),dstr(5:6),dstr(7:8) ,tstr(1:2),tstr(3:4),tstr(5:6)

   call map_write(map_new, trim(file_new))

end if

END PROGRAM main

!===============================================================================
